<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item;

use ApiPlatform\Core\DataProvider\ItemDataProviderInterface;
use ApiPlatform\Core\DataProvider\RestrictedDataProviderInterface;
use ApiPlatform\Exception\ItemNotFoundException;
use ApiPlatform\Exception\OperationNotFoundException;
use ApiPlatform\Exception\RuntimeException;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item\Assembler\RequestAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Item\Assembler\ResponseAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\Entity\EntityInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;

/**
 * @property string[] $classes
 */
class DataProvider implements ItemDataProviderInterface, RestrictedDataProviderInterface
{
    use LoggerAwareTrait;
    protected const DEFAULT_OPERATION_NAME = 'get';
    protected const PUT_OPERATION_NAME = 'put';
    protected const PATCH_OPERATION_NAME = 'patch';
    protected const DELETE_OPERATION_NAME = 'delete';

    public function __construct(
        private array $classes,
        private ActionInterface $action,
        private RequestAssemblerInterface $requestAssembler,
        private ResponseAssemblerInterface $responseAssembler,
        LoggerInterface $logger,
        private array $operationNames = [self::DEFAULT_OPERATION_NAME, self::PUT_OPERATION_NAME, self::PATCH_OPERATION_NAME, self::DELETE_OPERATION_NAME],
    ) {
        $this->setLogger($logger);
    }

    public function supports(string $resourceClass, string $operationName = null, array $context = []): bool
    {
        return in_array($resourceClass, $this->classes, true) && in_array($operationName, $this->operationNames, true);
    }

    public function getItem(string $resourceClass, $id, string $operationName = null, array $context = []): ?EntityInterface
    {
        $request = $this->requestAssembler->assemble($id);
        $response = $this->action->run($request);

        if ($response instanceof DefaultFailedResponse) {
            $this->logger->error('Cannot receive item: {id}', ['id' => $id]);

            throw match ($response->getStatus()) {
                Response::HTTP_BAD_REQUEST        => new BadRequestHttpException($response->getTitle()),
                Response::HTTP_UNAUTHORIZED       => new UnauthorizedHttpException($response->getTitle()),
                Response::HTTP_NOT_FOUND          => new ItemNotFoundException($response->getTitle()),
                Response::HTTP_METHOD_NOT_ALLOWED => new OperationNotFoundException($response->getTitle()),
                default                           => new RuntimeException($response->getTitle()),
            };
        }

        /** @noinspection PhpParamsInspection */
        return $this->responseAssembler->assemble($response);
    }
}
