<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection;

use ApiPlatform\Core\DataProvider\ContextAwareCollectionDataProviderInterface;
use ApiPlatform\Core\DataProvider\RestrictedDataProviderInterface;
use ApiPlatform\Exception\ItemNotFoundException;
use ApiPlatform\Exception\OperationNotFoundException;
use ApiPlatform\Exception\RuntimeException;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\PaginationResponseAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\RequestAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataProvider\Collection\Assembler\ResponseAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\Entity\EntityInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\PageOwnerInterface;

/**
 * @property string[] $classes
 */
class DataProvider implements ContextAwareCollectionDataProviderInterface, RestrictedDataProviderInterface
{
    use LoggerAwareTrait;

    protected const OPERATION_NAME = 'get';

    public function __construct(
        private array $classes,
        private ActionInterface $action,
        private RequestAssemblerInterface $requestAssembler,
        private ResponseAssemblerInterface $responseAssembler,
        private PaginationResponseAssemblerInterface $paginationResponseAssembler,
        LoggerInterface $logger,
        private array $operationNames = [self::OPERATION_NAME],
    ) {
        $this->setLogger($logger);
    }

    /**
     * @return EntityInterface[]
     */
    public function getCollection(string $resourceClass, string $operationName = null, array $context = []): iterable
    {
        $context['filters'] = $context['filters'] ?? [];
        $request = $this->requestAssembler->assemble($resourceClass, $operationName, $context);
        $response = $this->action->run($request);

        if ($response instanceof DefaultFailedResponse) {
            $this->logger->error('Cannot receive collection: {detail}', ['detail' => $response->getDetail()]);

            throw match ($response->getStatus()) {
                Response::HTTP_BAD_REQUEST        => new BadRequestHttpException($response->getTitle()),
                Response::HTTP_UNAUTHORIZED       => new UnauthorizedHttpException($response->getTitle()),
                Response::HTTP_NOT_FOUND          => new ItemNotFoundException($response->getTitle()),
                Response::HTTP_METHOD_NOT_ALLOWED => new OperationNotFoundException($response->getTitle()),
                default                           => new RuntimeException($response->getTitle()),
            };
        }

        /** @noinspection PhpParamsInspection */
        $items = $this->responseAssembler->assemble($response);

        if (!($response instanceof PageOwnerInterface)) {
            return $items;
        }

        if ($response->getItemsPerPage() > 0) {
            return $this->paginationResponseAssembler->assemble(
                $items,
                $response->getTotalItems(),
                $response->getPageIndex(),
                $response->getItemsPerPage(),
            );
        }

        return $this->paginationResponseAssembler->assemble(
            $items,
            count($items),
            0,
            count($items),
        );
    }

    public function supports(string $resourceClass, string $operationName = null, array $context = []): bool
    {
        return in_array($resourceClass, $this->classes, true) && in_array($operationName, $this->operationNames, true);
    }
}
