<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister;

use ApiPlatform\Exception\ItemNotFoundException;
use ApiPlatform\Exception\OperationNotFoundException;
use ApiPlatform\Exception\RuntimeException;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\Exception\UnauthorizedHttpException;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister\Assembler\RequestAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\DataPersister\Persister\Assembler\ResponseAssemblerInterface;
use XCart\Bundle\APIPlatformBridgeBundle\API\Entity\EntityInterface;
use XCart\Bundle\CommonBundle\DTO\Id\IdOwnerInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\PayloadOwnerResponseInterface;

class Persister implements PersisterInterface
{
    public function __construct(
        private ?ActionInterface $createAction,
        private ?ActionInterface $updateAction,
        private ?RequestAssemblerInterface $createRequestAssembler,
        private ?RequestAssemblerInterface $updateRequestAssembler,
        private ResponseAssemblerInterface $createResponseAssembler,
        private ResponseAssemblerInterface $updateResponseAssembler,
    ) {
    }

    public function persist(IdOwnerInterface $data, array $context): EntityInterface
    {
        /** @var PayloadOwnerResponseInterface $response */
        if ($this->isUpdate($data)) {
            $request  = $this->updateRequestAssembler->assemble($data, $context);
            $response = $this->updateAction->run($request);
        } else {
            $request  = $this->createRequestAssembler->assemble($data, $context);
            $response = $this->createAction->run($request);
        }

        if ($response instanceof DefaultFailedResponse) {
            $responseMessage = $response->getDetail() ?: $response->getTitle();

            throw match ($response->getStatus()) {
                Response::HTTP_BAD_REQUEST          => new BadRequestHttpException($responseMessage),
                Response::HTTP_UNAUTHORIZED         => new UnauthorizedHttpException($responseMessage),
                Response::HTTP_NOT_FOUND            => new ItemNotFoundException($responseMessage),
                Response::HTTP_METHOD_NOT_ALLOWED   => new OperationNotFoundException($responseMessage),
                Response::HTTP_UNPROCESSABLE_ENTITY => new UnprocessableEntityHttpException($responseMessage),
                default                             => new RuntimeException($responseMessage),
            };
        }

        return $this->isUpdate($data)
            ? $this->updateResponseAssembler->assemble($response)
            : $this->createResponseAssembler->assemble($response);
    }

    private function isUpdate(IdOwnerInterface $data): bool
    {
        return !empty($data->getId());
    }
}
