<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Transformer\XLite\Model\Product\Logic\Entity\Storefront;

use XCart\Logic\Entity\Storefront\ProductDimensions;
use XCart\Logic\Entity\Storefront\Image;
use XCart\Logic\Entity\Storefront\ProductCategory;
use XCart\Logic\Entity\Storefront\ProductAttribute;
use XCart\Logic\Entity\Storefront\ProductAttributeValue;
use XCart\Transformer\XLite\Model\AttributeValue\Logic\Entity\Storefront\AttributeValueInterface as AttributeValueTransformer;
use XLite\Model\Product as Source;
use XCart\Logic\Entity\Storefront\Product as Target;
use XCart\Factory\Logic\Entity\Storefront\ProductInterface as Factory;

class Product implements ProductInterface
{
    public function __construct(
        private Factory $factory,
        private AttributeValueTransformer $attributeValueTransformer
    ) {
    }

    public function transform(Source $input): Target
    {
        $target = $this->factory->create();

        $target->setId($input->getId());
        $target->setSku($input->getSku());
        $target->setName($input->getName());
        $target->setCleanUrl($input->getCleanURL());
        $target->setBriefDescription($input->getBriefDescription());
        $target->setPrice($input->getClearPrice());
        $target->setWeight($input->getWeight());
        $target->setRequiresShipping($input->getShippable());
        $target->setArrivalDate($input->getArrivalDate());
        $target->setAmount($input->getAmount());
        $target->setImages($this->getTransformedImages($input));
        $target->setDescription($input->getDescription());
        $target->setMetaTags($input->getMetaTags());
        $target->setMetaDescription($input->getMetaDesc());
        $target->setMetaTitle($input->getMetaTitle());
        $target->setPath($this->getTransformedPath($input));
        $target->setAttributes($this->getTransformedAttributes($input));

        $target->setDimensions(
            (new ProductDimensions())
                ->setWidth($input->getBoxWidth())
                ->setHeight($input->getBoxHeight())
                ->setLength($input->getBoxLength())
        );

        return $target;
    }

    /**
     * @return Image[]
     */
    private function getTransformedImages(Source $input): array
    {
        $transformedImages = [];

        /* @var \XLite\Model\Image\Product\Image $image */
        foreach ($input->getImages()->toArray() as $inputImage) {
            $transformedImages[] = (new Image())
                ->setUrl($inputImage->getFrontURL())
                ->setAlt($inputImage->getAlt())
                ->setWidth($inputImage->getWidth())
                ->setHeight($inputImage->getHeight());
        }

        return $transformedImages;
    }

    /**
     * @return ProductCategory[]
     */
    private function getTransformedPath(Source $input): array
    {
        $transformedPath = [];

        /* @var \XLite\Model\Category $category */
        foreach ($input->getCategory()->getPath() as $category) {
            $transformedPath[] = (new ProductCategory())
                ->setId($category->getId())
                ->setName($category->getName())
                ->setCleanUrl($category->getCleanURL());
        }

        return $transformedPath;
    }

    /**
     * @return ProductAttribute[]
     */
    private function getTransformedAttributes(Source $input): array
    {
        $transformedAttributes = [];

        /* @var \XLite\Model\Attribute $inputAttribute */
        foreach ($input->getVisibleAttributes() as $inputAttribute) {
            $inputAttributeGroup = $inputAttribute->getAttributeGroup();
            $isEditable          = false;

            $transformedAttribute = (new ProductAttribute())
                ->setId($inputAttribute->getId())
                ->setName($inputAttribute->getName())
                ->setType($inputAttribute->getType())
                ->setValues($this->getTransformedAttributeValues($input, $inputAttribute))
                ->setGroup($inputAttributeGroup ? $inputAttributeGroup->getName() : null);

            if ($inputAttribute->getType() === \XLite\Model\Attribute::TYPE_SELECT) {
                $transformedAttribute->setDisplayMode(
                    $inputAttribute->getDisplayMode($input)
                );
            }

            if ($inputAttribute->getType() === \XLite\Model\Attribute::TYPE_TEXT) {
                $isEditable = $inputAttribute->getAttributeValue($input)->getEditable();
                $transformedAttribute->setEditable($isEditable);
            }

            $displayAbove = ($isEditable || $inputAttribute->isMultiple($input))
                ? true
                : $inputAttribute->getDisplayAbove($input);

            $transformedAttribute->setDisplayAbove($displayAbove);

            $transformedAttributes[] = $transformedAttribute;
        }

        return $transformedAttributes;
    }

    /**
     * @return ProductAttributeValue[]
     */
    private function getTransformedAttributeValues(Source $input, \XLite\Model\Attribute $inputAttribute): array
    {
        $transformedAttributeValues = [];
        $inputAttributeValues       = $inputAttribute->getAttributeValue($input);

        if (!is_array($inputAttributeValues)) {
            $inputAttributeValues = [$inputAttributeValues];
        }

        /* @var \XLite\Model\AttributeValue\AAttributeValue $inputAttributeValue */
        foreach ($inputAttributeValues as $inputAttributeValue) {
            $transformedAttributeValues[] = $this->attributeValueTransformer->transform($inputAttributeValue, $input);
        }

        return $transformedAttributeValues;
    }
}
