<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Transformer\XLite\Model\Payment\Transaction\Logic\Entity\Storefront\Payment;

use XLite\Model\Payment\BackendTransaction;
use XLite\Model\Payment\BackendTransactionData;
use XLite\Model\Payment\Transaction as Source;
use XCart\Logic\Entity\Storefront\Payment\Action as Target;
use XCart\Factory\Logic\Entity\Storefront\Payment\ActionInterface as Factory;
use XLite\Model\Payment\TransactionData;

/**
 * Transfer from Doctrine Model to Logic Entity
 */
class Action implements ActionInterface
{
    public function __construct(
        private Factory $factory,
    ) {
    }

    public function transform(Source $input): Target
    {
        $target = $this->factory->create();

        $target->setId($input->getPublicTxnId());
        $target->setCartId($input->getOrder()->getPublicId());

        $target->setStatus($input->getStatus());
        $target->setValue($input->getValue());
        $target->setType($input->getType());
        $target->setData($this->getTransformedData($input));
        $target->setBackendTransactions($this->getTransformedBackendTransactions($input));

        $target->setProcessor($input->getPaymentMethod()?->getClass() ?: '');

        return $target;
    }

    /**
     * @return Target\TransactionData[]|null
     */
    private function getTransformedData(Source $input): ?array
    {
        /** @var TransactionData[] $data */
        $data = $input->getData();

        if (!$data) {
            return null;
        }

        $transformedData = [];
        foreach ($data as $item) {
            $transformedData[] = (new Target\TransactionData())
                ->setName($item->getName())
                ->setValue($item->getValue());
        }

        return $transformedData;
    }

    private function getTransformedBackendTransactions(Source $input): ?array
    {
        /** @var BackendTransaction[] $backendTransactions */
        $backendTransactions = $input->getBackendTransactions();

        if (!$backendTransactions) {
            return null;
        }

        $transformedBackendTransactions = [];
        foreach ($backendTransactions as $backendTransaction) {
            $transformedBackendTransactions[] = (new Target\BackendTransaction())
                ->setStatus($backendTransaction->getStatus())
                ->setValue($backendTransaction->getValue())
                ->setType($backendTransaction->getType())
                ->setData($this->getTransformedBackendTransactionsData($backendTransaction));
        }

        return $transformedBackendTransactions;
    }

    /**
     * @return Target\TransactionData[]|null
     */
    private function getTransformedBackendTransactionsData(BackendTransaction $backendTransaction): ?array
    {
        /** @var BackendTransactionData[] $data */
        $data = $backendTransaction->getData();

        if (!$data) {
            return null;
        }

        $transformedData = [];
        foreach ($data as $item) {
            $transformedData[] = (new Target\TransactionData())
                ->setName($item->getName())
                ->setValue($item->getValue());
        }

        return $transformedData;
    }
}
