<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Response;

use Symfony\Component\HttpFoundation\Response;
use XLite;
use XLite\Core\Layout;
use XLite\Core\Operator;
use XLite\Core\PreloadedLabels\Registrar;
use XLite\Core\Request;
use XLite\View\AView;
use XLite\View\CommonResources;
use XLite\View\Controller;

class XCartResponse extends Response
{
    private Request $request;

    private Operator $operator;

    private Layout $layout;

    private Registrar $registrar;

    private CommonResources $commonResources;

    /**
     * @throws \JsonException
     */
    public function __construct()
    {
        $this->request         = Request::getInstance();
        $this->operator        = Operator::getInstance();
        $this->layout          = Layout::getInstance();
        $this->registrar       = Registrar::getInstance();
        $this->commonResources = CommonResources::getInstance();

        parent::__construct($this->defineContent(), Response::HTTP_OK);

        foreach (XLite::getInstance()->getCookiesForHeaders() as $cookie) {
            $this->headers->setCookie($cookie);
        }
    }

    protected function defineContent(): string
    {
        $viewer = $this->getViewer();
        $viewer->init();

        if (!$this->request->isAJAX()) {
            $content = $this->getHtmlContent($viewer);
        } else {
            $content = $this->getAJAXContent($viewer);
        }

        return $content;
    }

    protected function getViewer(): AView
    {
        $class = $this->getViewerClass();

        return new $class($this->getViewerParams(), $this->getViewerTemplate());
    }

    protected function getViewerParams(): array
    {
        if (!$this->request->isAJAX()) {
            return [];
        }

        $data = $this->request->getData();
        unset($data['target'], $data['action']);

        return $data;
    }

    protected function getViewerTemplate(): string
    {
        return 'main.twig';
    }

    protected function getHtmlContent(AView $viewer): string
    {
        ob_start();
        $viewer->display();

        return ob_get_clean();
    }

    /**
     * @throws \JsonException
     */
    protected function getAJAXContent(AView $viewer): string
    {
        $id = $this->request->getUniqueIdentifier();

        $class = 'ajax-container-loadable'
            . ' ctrl-' . implode('-', $this->operator->getClassNameAsKeys(static::class))
            . ' widget-' . implode('-', $this->operator->getClassNameAsKeys($viewer));

        $content = $this->getAjaxResources() . PHP_EOL
            . $this->getAjaxPreloadedLabels() . PHP_EOL
            . $viewer->getContent();

        return "<div id=\"$id\" class=\"$class\">$content</div>";
    }

    /**
     * @throws \JsonException
     */
    protected function getAjaxResources(): string
    {
        $this->layout->registerResources([
            AView::RESOURCE_CSS => $this->commonResources->getCommonLessFiles(),
        ], 100, \XLite::INTERFACE_WEB, \XLite::ZONE_COMMON, 'getCommonFiles');

        if ($this->request->zone !== \XLite::ZONE_CUSTOMER && $this->layout->getZone() !== \XLite::ZONE_CUSTOMER) {
            $this->layout->registerResources([
                AView::RESOURCE_CSS => $this->commonResources->getCSSFiles()
            ], 10, null, null, 'getCSSFiles');
        }

        $resources = $this->layout->getRegisteredPreparedResources();

        $resContainer = [
            'widget' => $this->getViewerClass(),
        ];

        $js = $this->prepareResourcesList($resources[\XLite\View\AView::RESOURCE_JS]);
        $css = $this->prepareResourcesList($resources[\XLite\View\AView::RESOURCE_CSS]);

        if ($css || $js) {
            $resContainer = array_merge(
                $resContainer,
                [
                    'css' => $css,
                    'js' => $js,
                ]
            );
        }

        $resJson = json_encode($resContainer, JSON_THROW_ON_ERROR);

        return "<script type='application/json' data-resource>$resJson</script>";
    }

    /**
     * @throws \JsonException
     */
    protected function getAjaxPreloadedLabels(): string
    {
        $labels = $this->registrar->getRegistered();

        $labelsContainer = [
            'widget' => $this->getViewerClass(),
            'labels' => $labels
        ];

        $labelsJson = json_encode($labelsContainer, JSON_THROW_ON_ERROR | JSON_UNESCAPED_UNICODE);

        return "<script type='application/json' data-preloaded-labels>$labelsJson</script>";
    }

    protected function getViewerClass()
    {
        return $this->request->widget ?? Controller::class;
    }

    protected function prepareResourcesList($list): array
    {
        return array_map(static fn ($item) => $item['url'], array_values($list));
    }
}
