<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Repository;

use XLite\Core\Database;

trait TmpVarTrait
{
    protected $removedTmpVars = [];

    /**
     * Event task state prefix
     */
    public function getEventTaskStatePrefix(): string
    {
        return 'eventTaskState.';
    }

    /**
     * Set variable
     *
     * @param string  $name  Variable name
     * @param mixed   $value Variable value
     * @param boolean $flush Perform flush on return
     */
    public function setVar($name, $value, $flush = true)
    {
        $this->removeEntityFromUoW($name);
        $connection = Database::getEM()->getConnection();

        $query = 'REPLACE INTO ' . $this->getTableName()
            . ' set `name` = ?, `value` = ?';

        $connection->executeUpdate($query, [$name, serialize($value)]);
    }

    /**
     * Get variable
     *
     * @param string $name Variable name
     *
     * @return mixed
     */
    public function getVar($name)
    {
        $entity = $this->findOneBy(['name' => $name]);

        $value = $entity ? $entity->getValue() : null;

        if (!empty($value)) {
            $tmp = @unserialize($value, ['allowed_classes' => false]);
            if ($tmp !== false || $value === serialize(false)) {
                $value = $tmp;
            }
        }

        return $value;
    }

    /**
     * @param $name
     */
    public function removeVar($name)
    {
        $this->removeEntityFromUoW($name);
        $connection = Database::getEM()->getConnection();

        $query = 'DELETE FROM ' . $this->getTableName()
            . ' WHERE `name` = ?';

        $connection->executeUpdate($query, [$name]);
    }

    protected function removeEntityFromUoW($name)
    {
        if ($entity = Database::getRepo(\XLite\Model\TmpVar::class)->tryToFindEntityInIMByCriteria(['name' => $name])) {
            // See XCC-1038, garbage collector should not collect TmpVars which removed from identity map
            $this->removedTmpVars[] = $entity;
            Database::getEM()->getUnitOfWork()->removeFromIdentityMap($entity);
        }
    }

    // {{{ Event tasks-based temporary variable operations

    /**
     * Initialize event task state
     *
     * @param string $name    Event task name
     * @param array  $options Event options OPTIONAL
     *
     * @return array
     */
    public function initializeEventState($name, array $options = [])
    {
        $this->setEventState(
            $name,
            ['position' => 0, 'length' => 0, 'state' => \XLite\Core\EventTask::STATE_STANDBY] + $options
        );
    }

    /**
     * Get event task state
     *
     * @param string $name Event task name
     *
     * @return array
     */
    public function getEventState($name)
    {
        return $this->getVar($this->getEventTaskStatePrefix() . $name);
    }

    /**
     * Set event state
     *
     * @param string  $name  Event task name
     * @param array   $rec   Event task state
     * @param boolean $flush Flush task
     *
     * @return void
     */
    public function setEventState($name, array $rec, $flush = true)
    {
        $this->setVar($this->getEventTaskStatePrefix() . $name, $rec, $flush);
    }

    /**
     * Set event state
     *
     * @param string $name Event task name
     *
     * @return void
     */
    public function removeEventState($name)
    {
        $this->removeVar($this->getEventTaskStatePrefix() . $name);
    }

    /**
     * Check event state - finished or not
     *
     * @param string $name Event task name
     *
     * @return boolean
     */
    public function isFinishedEventState($name)
    {
        $record = $this->getEventState($name);

        return $record
            && ((int)$record['state'] === \XLite\Core\EventTask::STATE_FINISHED
                || (int)$record['state'] === \XLite\Core\EventTask::STATE_ABORTED);
    }

    /**
     * Check event state - finished or not
     *
     * @param string $name Event task name
     *
     * @return boolean
     */
    public function getEventStatePercent($name)
    {
        $percent = 0;

        $record = $this->getEventState($name);
        if ($record) {
            if ($this->isFinishedEventState($name)) {
                $percent = 100;
            } elseif (0 < $record['length']) {
                $percent = min(100, (int)($record['position'] / $record['length'] * 100));
            }
        }

        return $percent;
    }

    // }}}
}
