<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Operation\Service\Translation;

use Psr\Log\LoggerInterface;
use XLite\Core\Database;
use XLite\Model\Repo\ModelRepoTranslationTrait;

final class TranslationParser
{
    public function __construct(
        protected LoggerInterface $logger
    ) {
    }

    protected function getRawData(string $src): array
    {
        $data = [];

        $f = fopen($src, "rb");
        while (($row = fgetcsv($f)) !== false) {
            $data[] = $row;
        }

        fclose($f);

        return $data;
    }

    protected function parseData(array $data, array $modelsToInclude, string $lngCode): TranslationParsedData
    {
        $result = (new TranslationParsedData())->setLngCode($lngCode);

        $groupByModel = [];

        foreach ($data as $row) {
            if (
                count($row) < 4
                || ($row[0] !== $lngCode)
                || !in_array($row[1], $modelsToInclude, true)
            ) {
                $result->addIgnoredEntity($row);
            } else {
                $groupByModel[$row[1]][] = $row;
            }
        }

        foreach ($groupByModel as $modelFQN => $entities) {
            /** @var ModelRepoTranslationTrait|null $repo */
            $repo = Database::getRepo($modelFQN);

            if (
                $repo
                && method_exists($repo, 'generateYamlEntityForTranslationModule')
            ) {
                foreach ($entities as $entity) {
                    $modelFQN = $entity[1];

                    if ($yaml = $repo->generateYamlEntityForTranslationModule(array_slice($entity, 2), $lngCode)) {
                        if ($result->getModelEntitiesCount($modelFQN) === 0) {
                            $result->addDirective($modelFQN, $repo->getYamlDirectives());
                        }

                        $result->addYamlEntity($modelFQN, $yaml);
                    }
                }
            } else {
                $this->logger->warning(
                    (
                        $repo
                            ? "Cannot find 'generateYamlEntity' method in the " . get_debug_type($repo) . '. Entities ignored.'
                            : "Cannot find a repository class {$modelFQN}. Entities ignored."
                    ),
                    ['entities' => $entities]
                );

                $result->addIgnoredEntities($entities);
            }
        }

        return $result;
    }

    public function __invoke(
        string $csvFilename,
        array $modelsToInclude,
        string $lngCode
    ): TranslationParsedData {
        $rawData = $this->getRawData(
            $csvFilename
        );

        return $this->parseData(
            $rawData,
            $modelsToInclude,
            $lngCode ?: ($rawData[0][0] ?? '')
        );
    }
}
