<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Operation\Service\Translation;

final class TranslationParsedData
{
    protected int $yamlEntitiesCount = 0;

    protected array $coreData = [];

    protected array $modulesData = [];

    protected array $ignoredEntities = [];

    protected string $lngCode = '';

    public function getYamlEntitiesCount(): int
    {
        return $this->yamlEntitiesCount;
    }

    public function getIgnoredEntitiesCount(): int
    {
        return count($this->ignoredEntities);
    }

    public function setLngCode(string $lngCode): self
    {
        $this->lngCode = $lngCode;

        return $this;
    }

    public function getLngCode(): string
    {
        return $this->lngCode;
    }

    public function isCoreModel(string $modelFQN): bool
    {
        return (
            array_filter(
                explode('\\', $modelFQN)
            )[0] ?? ''
        ) === 'XLite';
    }

    public function getModuleIdByModelFQN(string $modelFQN): string
    {
        [$authorId, $nameId] = array_filter(
            explode('\\', $modelFQN)
        );

        return "{$authorId}-{$nameId}";
    }

    public function addDirective(string $modelFQN, array $directive): void
    {
        if ($this->isCoreModel($modelFQN)) {
            $this->coreData[$modelFQN][] = $directive;
        } else {
            $this->modulesData[$this->getModuleIdByModelFQN($modelFQN)][$modelFQN][] = $directive;
        }
    }

    public function addYamlEntity(string $modelFQN, array $entity): void
    {
        if ($this->isCoreModel($modelFQN)) {
            $this->coreData[$modelFQN][] = $entity;
        } else {
            $this->modulesData[$this->getModuleIdByModelFQN($modelFQN)][$modelFQN][] = $entity;
        }

        $this->yamlEntitiesCount++;
    }

    public function addIgnoredEntity(array $entity): void
    {
        $this->ignoredEntities[] = $entity;
    }

    public function addIgnoredEntities(array $entities): void
    {
        $this->ignoredEntities = array_merge(
            $this->ignoredEntities,
            $entities
        );
    }

    public function getCoreYamlData(): array
    {
        return $this->coreData;
    }

    protected function getCoreModelData(string $modelFQN): ?array
    {
        return $this->coreData[$modelFQN] ?? null;
    }

    public function getModuleYamlData(string $moduleId): ?array
    {
        return $this->modulesData[$moduleId] ?? null;
    }

    protected function getModuleModelData(string $modelFQN): ?array
    {
        return $this->modulesData[$this->getModuleIdByModelFQN($modelFQN)][$modelFQN] ?? null;
    }

    public function getModelData(string $modelFQN): ?array
    {
        return $this->isCoreModel($modelFQN)
            ? $this->getCoreModelData($modelFQN)
            : $this->getModuleModelData($modelFQN);
    }

    public function getModelEntitiesCount(string $modelFQN): int
    {
        if ($this->isCoreModel($modelFQN)) {
            return isset($this->coreData[$modelFQN]) ? count($this->coreData[$modelFQN]) : 0;
        }

        $moduleId = $this->getModuleIdByModelFQN($modelFQN);

        return isset($this->modulesData[$moduleId][$modelFQN]) ? count($this->modulesData[$moduleId][$modelFQN]) : 0;
    }

    /**
     * @return string[]
     */
    public function getModulesList(): array
    {
        return array_keys($this->modulesData);
    }

    public function isModulesTranslationsIncluded(): bool
    {
        return count($this->getModulesList()) > 0;
    }

    public function getIgnoredEntities(): array
    {
        return $this->ignoredEntities;
    }
}
