<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Operation\Service\Translation;

use SebastianBergmann\Template\RuntimeException;
use Symfony\Component\Yaml\Yaml;

final class TranslationCreateBase
{
    protected TranslationParsedData $data;

    protected TranslationCreateBaseConfig $config;

    protected function rmdir(string $dir): void
    {
        if (is_dir($dir)) {
            $objects = scandir($dir);

            foreach ($objects as $object) {
                if (!in_array($object, ['.', '..'])) {
                    $objectPath = $dir . '/' . $object;

                    if (is_dir($objectPath) && !is_link($objectPath)) {
                        $this->rmdir($objectPath);
                    } else {
                        unlink($objectPath);
                    }
                }
            }

            rmdir($dir);
        }
    }

    protected function getYamlCopyright(string $moduleId = null): string
    {
        $lngCode = strtoupper($this->data->getLngCode());

        $module = $moduleId ? "for {$moduleId} " : '';

        return "# vim: set ts=2 sw=2 sts=2 et:
#
# Translation data {$module}({$lngCode})
#
# Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
# See https://www.x-cart.com/license-agreement.html for license details.

";
    }

    protected function mkdir(string $dir): void
    {
        if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException("Cannot create directory {$dir}.");
        }

        if (!is_writable($dir)) {
            throw new RuntimeException("Cannot write to {$dir}.");
        }
    }

    protected function getYamlModules(TranslationParsedData $data): array
    {
        return array_filter(
            array_keys($yamlData),
            static fn (string $key): bool => $key !== 'core'
        );
    }

    protected function isYamlGeneratedForModules(array $yamlData): bool
    {
        return !empty($this->getYamlModules($yamlData));
    }

    protected function getYamlHooksData(): string
    {
        $moduleId = $this->config->getModuleId();

        $moduleRebuildHookFQN = str_replace('-', '\\', $moduleId) . '\\LifetimeHook\\Rebuild';

        return '#
# Module services
#
# Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
# See https://www.x-cart.com/license-agreement.html for license details.

'
            . Yaml::dump(
                [
                    'services' => [
                        'defaults'            => [
                            '_autowire' => true,
                        ],
                        $moduleRebuildHookFQN => [
                            'tags' => [
                                [
                                    'name'     => 'xcart.lifetime-hook',
                                    'moduleId' => $moduleId,
                                    'type'     => 'rebuild',
                                    'method'   => 'onRebuild',
                                ]
                            ]
                        ]
                    ]
                ],
                2,
                2
            );
    }

    protected function getModuleFixturesFileRelativePath(string $moduleId): string
    {
        return "/resources/hooks/rebuild/{$moduleId}.yaml";
    }

    protected function getRebuildHookCode(): string
    {
        $hookNamespace = str_replace('-', '\\', $this->config->getModuleId()) . '\\LifetimeHook';

        $result = "<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace {$hookNamespace};

use XCart\Doctrine\FixtureLoader;
use XCart\Domain\ModuleManagerDomain;

final class Rebuild
{
    public function __construct(
        private ModuleManagerDomain \$moduleManager,
        private FixtureLoader \$fixtureLoader
    ) {
    }

    public function onRebuild(): void
    {";

        foreach ($this->data->getModulesList() as $moduleId) {
            $moduleFixturesPath = str_replace('-', '/', $moduleId) . $this->getModuleFixturesFileRelativePath($moduleId);

            $result .= "
        if (\$this->moduleManager->isEnabled('{$moduleId}')) {
            \$this->fixtureLoader->loadYaml(LC_DIR_MODULES . '{$moduleFixturesPath}');
        }";
        }

        return $result . "
    }
}
";
    }

    protected function writeData(): TranslationCreateBaseResult
    {
        $result = new TranslationCreateBaseResult();
        $moduleDir = $this->config->getModuleBaseDir();

        if (file_exists($moduleDir) && is_dir($moduleDir)) {
            $this->rmdir($moduleDir);
        }

        $moduleDir .= '/' . $this->config->getModuleSubdir();
        $result->setModuleDir($moduleDir);
        $this->mkdir($moduleDir . '/config');

        if ($this->data->isModulesTranslationsIncluded()) {
            $this->mkdir($moduleDir . '/resources/hooks/rebuild');
            $this->mkdir($moduleDir . '/config/services');
            $this->mkdir($moduleDir . '/src/LifetimeHook');

            file_put_contents(
                $moduleDir . '/config/services/hooks.yaml',
                $this->getYamlHooksData()
            );

            file_put_contents(
                $moduleDir . '/src/LifetimeHook/Rebuild.php',
                $this->getRebuildHookCode()
            );

            foreach ($this->data->getModulesList() as $moduleId) {
                file_put_contents(
                    $moduleDir . $this->getModuleFixturesFileRelativePath($moduleId),
                    $this->getYamlCopyright($moduleId)
                        . Yaml::dump(
                            $this->data->getModuleYamlData($moduleId),
                            2,
                            2
                        )
                );
            }
        }

        file_put_contents(
            $moduleDir . '/config/install.yaml',
            $this->getYamlCopyright()
                . Yaml::dump(
                    $this->data->getCoreYamlData(),
                    2,
                    2
                )
        );

        if ($this->data->getIgnoredEntitiesCount() > 0) {
            $ignoredLabelsFilePath = $this->config->getRootDir() . '/' . $this->config->getIgnoredLabelsFilename();

            $f = fopen(
                $this->config->getRootDir() . '/' . $this->config->getIgnoredLabelsFilename(),
                'wb'
            );

            foreach ($this->data->getIgnoredEntities() as $entity) {
                fputcsv($f, $entity);
            }

            fclose($f);

            $result->setIgnoredLabelsFilePath($ignoredLabelsFilePath);
        }

        return $result;
    }

    public function __invoke(TranslationCreateBaseConfig $config): TranslationCreateBaseResult
    {
        $this->config = $config;
        $this->data = $config->getData();

        return $this->writeData();
    }
}
