<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Notification;

use Doctrine\ORM\EntityManagerInterface;
use XLite\Core\Auth;
use XLite\Model\Config;

class InfoBlockManager
{
    private Config $showedNotificationsConfig;

    public function __construct(
        private EntityManagerInterface $entityManager,
        private Auth $auth
    ) {
        $this->defineShowedNotificationsConfig();
    }

    /**
     * @throws \JsonException
     */
    public function closeNotification(string $notificationType): void
    {
        $timestamp = time();
        $showedNotifications = $this->getShowedNotifications();
        $profileId = $this->getProfileId();

        $showedNotifications[$profileId][$notificationType] = [
            'ReadTimestamp' => $timestamp,
            'UpdateTimestamp' => $timestamp
        ];
        $this->showedNotificationsConfig->setValue(
            json_encode($showedNotifications, JSON_THROW_ON_ERROR)
        );

        $this->entityManager->flush();
    }

    public function getLastUpdateTimestamp(string $notificationType): int
    {
        $profileId = $this->getProfileId();
        $result = $this->getShowedNotifications()[$profileId][$notificationType]['UpdateTimestamp'] ?? null;

        if ($result === null) {
            $this->setLastUpdateTimestamp($notificationType, time());
        }

        return $this->getShowedNotifications()[$profileId][$notificationType]['UpdateTimestamp'];
    }

    public function getLastReadTimestamp(string $notificationType): int
    {
        $profileId = $this->getProfileId();

        return (int) ($this->getShowedNotifications()[$profileId][$notificationType]['ReadTimestamp'] ?? 0);
    }

    public function setLastUpdateTimestamp(string $notificationType, int $timestamp): void
    {
        $showedNotification = $this->getShowedNotifications();
        $profileId = $this->getProfileId();

        $showedNotification[$profileId][$notificationType]['UpdateTimestamp'] = $timestamp;
        $this->showedNotificationsConfig->setValue(json_encode($showedNotification, JSON_THROW_ON_ERROR));

        $this->entityManager->flush();
    }

    public function getNotificationTypes(): array
    {
        $showedNotifications = $this->getShowedNotifications()[$this->getProfileId()] ?? [];

        return array_keys($showedNotifications);
    }

    public function getClosedNotificationTypes(): array
    {
        $showedNotifications = $this->getShowedNotifications()[$this->getProfileId()] ?? [];

        $showedNotifications = array_filter(
            $showedNotifications,
            static fn (array $notification): bool => $notification['UpdateTimestamp'] <= ($notification['ReadTimestamp'] ?: 0)
        );

        return array_keys($showedNotifications);
    }

    private function defineShowedNotificationsConfig(): void
    {
        $this->showedNotificationsConfig = $this->entityManager
            ->getRepository(Config::class)
            ->findOneBy([
                'category' => 'Internal',
                'name' => 'showed_notifications'
            ]);
    }

    /**
     * @throws \JsonException
     */
    private function getShowedNotifications(): array
    {
        return json_decode($this->showedNotificationsConfig->getValue(), true, 512, JSON_THROW_ON_ERROR);
    }

    private function getProfileId(): string
    {
        return (string) $this->auth->getProfile()?->getProfileId();
    }
}
