<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Monolog\Handler;

use ErrorException;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;

use function date;

class XCFileHandler extends StreamHandler
{
    /**
     * @var array
     */
    private $streams = [];

    public function __construct(
        private string $path,
        private string $file,
        private bool $skipDeprecation,
        int $level = Logger::DEBUG,
    ) {
        parent::__construct($this->path, $level, true, 0644, false);
    }

    protected function write(array $record): void
    {
        if (empty($this->file)) {
            return;
        }

        if (
            $this->skipDeprecation
            && isset($record['context']['exception'])
            && $record['context']['exception'] instanceof ErrorException
            && ($record['context']['exception']->getSeverity() & (\E_DEPRECATED | \E_USER_DEPRECATED))
        ) {
            return;
        }

        $streamName = !empty($record['context']['_stream']) ? $record['context']['_stream'] : $this->file;
        unset($record['context']['_stream']);

        $this->setStreamName($streamName);

        parent::write($record);

        $this->streams[$streamName]['url']    = $this->url;
        $this->streams[$streamName]['stream'] = $this->stream;
    }

    private function createFilePath(string $streamName): string
    {
        return $this->path . '/' . date('Y/m') . '/' . $streamName . '.' . date('Y-m-d') . '.log';
    }

    private function setStreamName($streamName): void
    {
        $url = $this->createFilePath($streamName);

        if (isset($this->streams[$streamName]['url']) && $this->streams[$streamName]['url'] === $url) {
            $this->url    = $this->streams[$streamName]['url'];
            $this->stream = $this->streams[$streamName]['stream'] ?? null;
        } else {
            $this->url    = $url;
            $this->stream = null;
        }
    }
}
