<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Logic\Action\Storefront\ResetPasswordAction\ResetPassword;

use ApiPlatform\Exception\RuntimeException;
use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use Throwable;
use XCart\Bundle\DoctrineBridgeBundle\Repository\ReadRepositoryInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\EmptySuccessfulResponse;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\ResetPasswordAction\CreateOne\DTO\Request;
use XLite\Core\Auth;

class Action implements ActionInterface
{
    public function __construct(
        private ReadRepositoryInterface $profileRepository,
    ) {
    }

    /**
     * @param Request $request
     *
     * @return EmptySuccessfulResponse
     */
    public function run(RequestInterface $request): ResponseInterface
    {
        try {
            $username = $request->getPayload()->getUsername();

            if (!$username) {
                return (new DefaultFailedResponse())
                    ->setTitle('Field "username" is required')
                    ->setStatus(HTTPResponse::HTTP_BAD_REQUEST);
            }

            $profile = $this->profileRepository->findOneBy(['login' => $username]);

            if (!$profile) {
                return (new DefaultFailedResponse())
                    ->setTitle('There is no user with the specified email address')
                    ->setStatus(HTTPResponse::HTTP_BAD_REQUEST);
            }

            $password = \XLite\Core\Database::getRepo('XLite\Model\Profile')->generatePassword();
            $profile->setPassword(Auth::encryptPassword($password));
            $result = $profile->update();

            if (!$result) {
                throw new RuntimeException('Cannot set new password');
            }

            \XLite\Core\Mailer::resetPassword($profile, $password);
        } catch (Throwable $e) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot assemble entity: %s', $e))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        return new EmptySuccessfulResponse();
    }
}
