<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\Product\GetOne;

use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\Product\GetOne\Assembler\QueryBuilder\FindOneQueryBuilderAssemblerInterface;
use XCart\Logic\Action\Storefront\Product\GetOne\DTO\Response as GetOneProductResponse;
use XCart\Logic\Entity\Storefront\Product as ProductLogicDto;
use XCart\Logic\Entity\Storefront\ProductCategory as ProductCategoryLogicDto;
use XLite\Model\Product as ProductModel;
use XLite\Model\Repo\Category as CategoryRepo;

class ProductPathActionDecorator implements ActionInterface
{
    public function __construct(
        private ActionInterface $inner,
        private FindOneQueryBuilderAssemblerInterface $queryBuilderAssembler,
        private CategoryRepo $categoryRepository,
    ) {
    }

    public function run(RequestInterface $request): ResponseInterface
    {
        $response = $this->inner->run($request);

        if ($response instanceof GetOneProductResponse) {
            $membershipId = $request->getProfile()?->getMembershipId();

            /** @var ProductLogicDto $payload */
            $payload = $response->getPayload();

            $qb           = $this->queryBuilderAssembler->assemble($request);
            $productModel = $qb->getQuery()->getOneOrNullResult();

            $path = [];

            foreach ($this->getPath($productModel) as $category) {
                if (
                    !$category->getEnabled()
                    || (
                        $category->getMemberships()->count() > 0
                        && !in_array($membershipId, $category->getMembershipIds(), true)
                    )
                ) {
                    break;
                }

                $path[] = (new ProductCategoryLogicDto())
                    ->setId($category->getId())
                    ->setName($category->getName())
                    ->setCleanUrl($category->getCleanURL());
            }

            $payload->setPath($path);
        }

        return $response;
    }

    /**
     * @return \XLite\Model\Category[]
     */
    private function getPath(ProductModel $product): array
    {
        $productCategory = $product->getCategory();

        return $this->categoryRepository
            ->createPureQueryBuilder('c')
            ->linkLeft('c.memberships', 'm')
            ->andWhere('c.lpos <= :lpos')
            ->andWhere('c.rpos >= :rpos')
            ->andWhere('c.depth >= 0')
            ->orderBy('c.lpos', 'ASC')
            ->setParameter('lpos', $productCategory->getLpos())
            ->setParameter('rpos', $productCategory->getRpos())
            ->getResult();
    }
}
