<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\Order\CreateOne;

use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\Order\CreateOne\DTO\Request;
use XCart\Logic\Action\Storefront\Order\CreateOne\DTO\Response;
use XCart\Logic\Entity\Storefront\NewOrder;
use XLite\Model\Cart as CartModel;
use XLite\Model\Repo\Cart as CartRepository;
use XLite\Model\Repo\Shipping\Method as ShippingMethodRepository;

class Action implements ActionInterface
{
    public function __construct(
        private CartRepository $cartWriteRepository,
        private ShippingMethodRepository $shippingMethodReadRepository,
    ) {
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function run(RequestInterface $request): ResponseInterface
    {
        /** @var CartModel $cart */
        $cart = $this->cartWriteRepository->find(
            $request->getPayload()->getCartId()
        );

        if (!$cart->getProfile()->getEmail()) {
            return (new DefaultFailedResponse())
                ->setType('E-mail is missed')
                ->setTitle('E-mail is missed, can\'t place order')
                ->setStatus(HTTPResponse::HTTP_BAD_REQUEST);
        }

        $this->setShippingMethodName($cart);

        if (!$cart->tryClose()) {
            return (new DefaultFailedResponse())
                ->setType('In progress')
                ->setTitle('Transaction is in progress, can\'t place order')
                ->setStatus(HTTPResponse::HTTP_BAD_REQUEST);
        }

        return (new Response())
            ->setPayload(
                (new NewOrder())
                    ->setId($cart->getOrderId())
                    ->setCartPublicId($cart->getPublicId())
                    ->setNumber($cart->getOrderNumber() ?: '')
            );
    }

    private function setShippingMethodName(CartModel $cart): void
    {
        $shippingMethod = $this->shippingMethodReadRepository->find(
            $cart->getShippingId()
        );

        $cart->setShippingMethodName($shippingMethod?->getName());
    }
}
