<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\Category\GetOne;

use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use Throwable;
use XCart\Bundle\DoctrineBridgeBundle\Assembler\GetOne\Response\ResponseAssemblerInterface;
use XCart\Bundle\DoctrineBridgeBundle\Repository\ReadRepositoryInterface;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\Category\GetOne\Assembler\QueryBuilder\FindOneQueryBuilderAssemblerInterface;
use XLite\Model\Category;
use XLite\Model\Profile;

class Action implements ActionInterface
{
    public function __construct(
        private FindOneQueryBuilderAssemblerInterface $queryBuilderAssembler,
        private ReadRepositoryInterface $categoryRepository,
        private ResponseAssemblerInterface $responseAssembler,
    ) {
    }

    /**
     * @param \XCart\Logic\Action\Storefront\Category\GetOne\DTO\Request $request
     */
    public function run(RequestInterface $request): ResponseInterface
    {
        try {
            $qb       = $this->queryBuilderAssembler->assemble($request);
            $category = $qb->getQuery()->getOneOrNullResult();
        } catch (Throwable $e) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle(sprintf('Cannot read entity from DB: %s', $e))
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (
            !$category
            || $this->hasUnavailableParentCategories($category, $request->getProfile())
        ) {
            return (new DefaultFailedResponse())
                ->setType('Not found')
                ->setTitle('Cannot find entity')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        /** @noinspection PhpIncompatibleReturnTypeInspection */
        return $this->responseAssembler->assemble($category);
    }

    private function hasUnavailableParentCategories(Category $category, ?Profile $profile): bool
    {
        $qb = $this->categoryRepository
            ->createPureQueryBuilder('c')
            ->select('count(c.category_id)')
            ->linkLeft('c.memberships', 'm')
            ->andWhere('c.lpos < :lpos')
            ->andWhere('c.rpos > :rpos')
            ->andWhere('c.enabled = 1')
            ->setParameter('lpos', $category->getLpos())
            ->setParameter('rpos', $category->getRpos());

        if ($membershipId = $profile?->getMembershipId()) {
            $qb->andWhere('m.membership_id = :membershipId OR m.membership_id IS NULL')
                ->setParameter('membershipId', $membershipId);
        } else {
            $qb->andWhere('m.membership_id IS NULL');
        }

        $availableParentsCount = (int) $qb
            ->getQuery()
            ->getSingleScalarResult();

        $allParentsCount = ($category->getDepth() + 1);

        return $availableParentsCount !== $allParentsCount;
    }
}
