<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\CartMerge\CreateOne;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\Cart\MergeRequest\CreateOne\DTO\Request;
use XCart\Logic\Action\Storefront\Cart\MergeRequest\CreateOne\DTO\Response;
use XCart\Logic\Entity\Storefront\Cart\NewMergeRequest;
use XLite\Model\Cart as CartModel;
use XLite\Model\Repo\Cart as CartRepository;
use XLite\Model\Repo\Profile as ProfileAlias;

class Action implements ActionInterface
{
    public function __construct(
        private CartRepository $cartWriteRepository,
        private ProfileAlias $profileWriteRepository,
        private EntityManagerInterface $entityManager,
    ) {
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function run(RequestInterface $request): ResponseInterface
    {
        /** @var ?CartModel $cart */
        $cart = $this->cartWriteRepository->findOneBy(['public_id' => $request->getPayload()->getAnonymousCartId()]);
        if (!$cart) {
            return (new DefaultFailedResponse())
                ->setType('Not Found')
                ->setTitle('Cannot find anonymous cart')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        if ($cart->getOrigProfile() && !$cart->getOrigProfile()->getAnonymous()) {
            return (new DefaultFailedResponse())
                ->setType('Not Found')
                ->setTitle('Cart not anonymous')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        $this->profileWriteRepository->delete($cart->getProfile(), false);

        /** @var ?CartModel $cart */
        $currentCart = $this->cartWriteRepository->findOneBy(['public_id' => $request->getPayload()->getId()]);
        if ($currentCart) {
            $currentCart->merge($cart);

            $cart->setProfile(null);
            $cart->setOrigProfile(null);
            $this->cartWriteRepository->delete($cart, false);
        } else {
            $profile = $this->profileWriteRepository->find($request->getPayload()->getProfileId());
            $cart->setProfile($profile);
            $cart->setOrigProfile($profile);
        }

        $this->entityManager->flush();

        return (new Response())
            ->setPayload(
                (new NewMergeRequest())
                    ->setId($currentCart->getPublicId())
                    ->setAnonymousCartId($request->getPayload()->getAnonymousCartId())
            );
    }
}
