<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\CartItem\Common\Assembler;

use ApiPlatform\Core\Exception\InvalidArgumentException;
use Doctrine\ORM\EntityManagerInterface;
use XLite\Model\Attribute;
use Xlite\Model\AttributeValue\AAttributeValue;
use XLite\Model\AttributeValue\AttributeValueText;
use XLite\Model\OrderItem as Target;
use XCart\Logic\Entity\Storefront\Cart\Item as Source;
use XCart\Logic\Entity\Storefront\Cart\NewItem as NewSource;
use XLite\Model\OrderItem\AttributeValue;
use XLite\Model\Product as ProductModel;

class ModelAttributeValuesAssembler implements ModelAttributeValuesAssemblerInterface
{
    public function __construct(
        private EntityManagerInterface $entityManager,
    ) {
    }

    public function assemble(Target $target, ProductModel $product, Source|NewSource $input): void
    {
        foreach ($target->getAttributeValues() as $av) {
            $this->entityManager->remove($av);
        }

        $target->getAttributeValues()->clear();

        foreach ($product->getEditableAttributes() as $attribute) {
            $found = false;
            foreach ($input->getAttributes() as $inputAttributeValue) {
                if ($inputAttributeValue->getAttributeId() === $attribute->getId()) {
                    $found = $this->setAttribute($target, $inputAttributeValue, $attribute, $product);
                    break;
                }
            }

            if (!$found) {
                $this->setDefaultAttribute($target, $product, $attribute);
            }
        }
    }

    private function setAttribute(Target $target, Source\AttributeValue $inputAttributeValue, Attribute $attribute, ProductModel $product): bool
    {
        if ($attribute->getType() === Attribute::TYPE_TEXT) {
            $newValue = $this->setTextAttribute($inputAttributeValue, $attribute, $product);
            $newValue->setOrderItem($target);
            $target->addAttributeValues($newValue);

            return true;
        }

        foreach ($attribute->getAttributeValues() as $attributeValue) {
            if ($attributeValue->getId() === $inputAttributeValue->getAttributeValueId()) {
                $newValue = $this->setSelectableAttribute($inputAttributeValue, $attribute, $attributeValue);
                $newValue->setOrderItem($target);
                $target->addAttributeValues($newValue);

                return true;
            }
        }

        throw new InvalidArgumentException(sprintf('Attribute #%d has wronmg value #%d', $attribute->getId(), $inputAttributeValue->getAttributeValueId()));
    }

    private function setTextAttribute(Source\AttributeValue $inputAttributeValue, Attribute $attribute, ProductModel $product): AttributeValue
    {
        $defaultValue = $attribute->getAttributeValue($product);

        $newValue = new AttributeValue();
        $newValue->setName($attribute->getName());
        $newValue->setValue($inputAttributeValue->getAttributeValue());
        $newValue->setAttributeId($attribute->getId());
        $newValue->setAttributeValue($defaultValue);

        return $newValue;
    }

    private function setSelectableAttribute(Source\AttributeValue $inputAttributeValue, Attribute $attribute, AAttributeValue $attributeValue): AttributeValue
    {
        $newValue = new AttributeValue();
        $newValue->setName($attribute->getName());
        $newValue->setValue($inputAttributeValue->getAttributeValue() ?? $attributeValue->getName());
        $newValue->setAttributeId($attribute->getId());
        $newValue->setAttributeValue($attributeValue);

        return $newValue;
    }

    private function setDefaultAttribute(Target $target, ProductModel $product, Attribute $attribute): AttributeValue
    {
        if ($attribute->getType() === Attribute::TYPE_TEXT) {
            $defaultValue = $attribute->getAttributeValue($product);
        } else {
            $defaultValue = $attribute->getDefaultAttributeValue($product);
        }

        $newValue = new AttributeValue();
        $newValue->setName($attribute->getName());
        $newValue->setValue($defaultValue instanceof AttributeValueText ? $defaultValue->getValue() : $defaultValue->getName());
        $newValue->setAttributeId($attribute->getId());

        $newValue->setOrderItem($target);
        $target->addAttributeValues($newValue);

        $newValue->setAttributeValue($defaultValue);

        return $newValue;
    }
}
