<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Logic\Action\Storefront\CartCalculatedShippingRates\CreateOne;

use Symfony\Component\HttpFoundation\Response as HTTPResponse;
use XCart\Bundle\LogicBundle\Action\ActionInterface;
use XCart\Bundle\LogicBundle\DTO\Request\RequestInterface;
use XCart\Bundle\LogicBundle\DTO\Response\DefaultFailedResponse;
use XCart\Bundle\LogicBundle\DTO\Response\ResponseInterface;
use XCart\Logic\Action\Storefront\Cart\CalculatedShippingRates\CreateOne\DTO\Request;
use XCart\Logic\Action\Storefront\Cart\CalculatedShippingRates\CreateOne\DTO\Response;
use XCart\Logic\Entity\Storefront\Cart\NewCalculatedShippingRates;
use XCart\Logic\Entity\Storefront\Cart\ShippingMethod;
use XCart\Logic\Entity\Storefront\Cart\ShippingRate;
use Xlite\Logic\Order\Modifier\Shipping;
use Xlite\Logic\Order\Modifier\Shipping as ShippingModifier;
use XLite\Model\Base\Surcharge;
use XLite\Model\Cart as CartModel;
use XLite\Model\Order\Modifier as OrderModifier;
use XLite\Model\Repo\Cart as CartRepository;

class Action implements ActionInterface
{
    public function __construct(
        private CartRepository $cartReadRepository,
    ) {
    }

    /**
     * @param Request $request
     *
     * @return Response
     */
    public function run(RequestInterface $request): ResponseInterface
    {
        $entity = $request->getPayload();

        /** @var ?CartModel $cart */
        $cart = $this->cartReadRepository->find($entity->getCartId());
        $failedResponse = $this->validateCart($cart, $entity);
        if ($failedResponse) {
            return $failedResponse;
        }

        /** @var ?OrderModifier $modifier */
        $modifier = $cart->getModifier(Surcharge::TYPE_SHIPPING, 'SHIPPING');
        $failedResponse = $this->validateShippingModifier($modifier);
        if ($failedResponse) {
            return $failedResponse;
        }

        return (new Response())
            ->setPayload(
                (new NewCalculatedShippingRates())
                    ->setId($entity->getCartId())
                    ->setCartId($entity->getCartId())
                    ->setRates($this->buildShippingRates($modifier->getModifier()))
            );
    }

    private function validateCart(?CartModel $cart, NewCalculatedShippingRates $entity): ?DefaultFailedResponse
    {
        if (!$cart) {
            return (new DefaultFailedResponse())
                ->setType('Not Found')
                ->setTitle('Cannot found cart')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        return null;
    }

    private function validateShippingModifier(?OrderModifier $shippingModifier): ?DefaultFailedResponse
    {
        if (!$shippingModifier) {
            return (new DefaultFailedResponse())
                ->setType('Not Found')
                ->setTitle('Shipping disabled')
                ->setStatus(HTTPResponse::HTTP_NOT_FOUND);
        }

        if (!($shippingModifier->getModifier() instanceof ShippingModifier)) {
            return (new DefaultFailedResponse())
                ->setType('Internal error')
                ->setTitle('Shipping internal modifier error')
                ->setStatus(HTTPResponse::HTTP_INTERNAL_SERVER_ERROR);
        }

        return null;
    }

    /**
     * @return list<ShippingRate>
     */
    public function buildShippingRates(Shipping $modifier): array
    {
        $shippingRates = [];
        foreach ($modifier->getRates() as $rate) {
            $shippingRates[] = (new ShippingRate())
                ->setMethod((new ShippingMethod())->setId($rate->getMethodId())->setName($rate->getMethodName()))
                ->setValue($rate->getTotalRate())
                ->setPreparedDeliveryTime($rate->getPreparedDeliveryTime());
        }

        return $shippingRates;
    }
}
