<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\LifetimeHook\Upgrade;

use XCart\Doctrine\FixtureLoader;
use XLite\Core\Database;
use XLite\Model\AddressField;
use XLite\Model\Config;
use XLite\Model\Role\Permission;

final class UpgradeTo551build0
{
    protected array $config;

    public function __construct(
        private FixtureLoader $fixtureLoader
    ) {
    }

    /**
     * @param array $config
     */
    public function setConfig(array $config): void
    {
        $this->config = $config;
    }

    public function onUpgrade(): void
    {
        $this->fixtureLoader->loadYaml(LC_DIR_ROOT . 'upgrade/5.5/1.0/upgrade.yaml');
        $this->updatePermissions();
        $this->removeDuplicateAddressFieldValues();
        $this->removeInternalCronEnabled();

        /** @var AddressField|null $streetAddressField */
        $streetAddressField = Database::getRepo('XLite\Model\AddressField')->findOneBy([
            'serviceName' => 'street'
        ]);

        if ($streetAddressField) {
            $streetAddressField->setServiceName('address1');
            $address1Position = $streetAddressField->getPosition();
            Database::getEM()->flush();

            /** @var AddressField[] $addressFields */
            $addressFields = Database::getRepo('XLite\Model\AddressField')->findAll();
            $isAddressFieldsPositionSet = array_reduce(
                $addressFields,
                static fn(bool $carry, AddressField $field): bool => $carry || ($field->getPosition() !== 0),
                false
            );

            if (!$isAddressFieldsPositionSet) {
                $address1Position = $streetAddressField->getId() * 10;
            }

            foreach ($addressFields as $field) {
                $serviceName = $field->getServiceName();

                if ($serviceName === 'address2') {
                    $field->setPosition($address1Position + 10);
                } elseif ($serviceName === 'address3') {
                    $field->setPosition($address1Position + 20);
                } elseif (
                    $isAddressFieldsPositionSet
                    && $field->getPosition() >= $address1Position
                    && $serviceName !== 'address1'
                ) {
                    $field->setPosition($field->getPosition() + 20);
                } elseif (!$isAddressFieldsPositionSet) {
                    $field->setPosition($field->getId() * 10);
                }
            }

            Database::getEM()->flush();
        }
    }

    private function updatePermissions(): void
    {
        /** @var Permission[]|null $permissions */
        $permissions = Database::getRepo(Permission::class)?->findAll();

        foreach ($permissions as $permission) {
            $oldCode = $permission->getCode();
            $newCode = str_starts_with($oldCode, 'ROLE_')
                ? $oldCode
                : 'ROLE_' . strtoupper(str_replace([' ', '[', ']'], ['_', '', ''], $oldCode));
            $permission->setCode($newCode);
        }

        Database::getEM()->flush();
    }

    private function removeDuplicateAddressFieldValues(): void
    {
        $tablePrefix = '';
        foreach ($this->config as $data) {
            if ($data['class'] === '*') {
                $tablePrefix = $data['prefix'];
                break;
            }
        }

        if ($tablePrefix !== '') {
            $tablePrefix .= '_';
        }

        $dbal = Database::getEM()->getConnection();
        $tableName = $tablePrefix . 'address_field_value';

        $addressValueIds = $dbal->fetchAllAssociative(
            "SELECT 
                {$tableName}.`id`
            FROM
                {$tableName}
            LEFT OUTER JOIN 
                (
                    SELECT MIN(`id`) AS `id`, `address_field_id`, `address_id`
                    FROM {$tableName}
                    GROUP BY `address_field_id`, `address_id`
                ) AS `tmp` 
            ON 
                {$tableName}.`id` = `tmp`.`id`  
            WHERE
                `tmp`.`id` IS NULL"
        );
        foreach ($addressValueIds as $addressValueItem) {
            $dbal->delete($tableName, ['id' => $addressValueItem['id']]);
        }
    }

    private function removeInternalCronEnabled(): void
    {
        /** @var \XLite\Model\Repo\Config|null $repo */
        if ($repo = Database::getRepo(Config::class)) {
            $model = $repo->findOneBy([
                'category' => 'General',
                'name' => 'internal_cron_enabled'
            ]);

            if ($model) {
                Database::getEM()->remove($model);
                Database::getEM()->flush();
            }
        }
    }
}
