<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\EventListener\Payment;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use XCart\Event\Payment\PaymentActionEvent;
use XCart\Payment\RequestProcessor;
use XLite\Core\Mailer;
use XLite\Core\OrderHistory;
use XLite\Core\Translation;

final class CommonProcess implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    public function __construct(
        private Translation $translation,
        private OrderHistory $orderHistory
    ) {
    }

    public function onPaymentActionPreProcess(PaymentActionEvent $event): void
    {
        $this->logger->debug(
            $event->getProcessorServiceName() . ' payment gateway : return',
            [
                'action'    => $event->getAction(),
                'data'      => $event->getData(),
                'meta_data' => $event->getMetaData(),
            ]
        );
    }

    public function onPaymentActionPostProcess(PaymentActionEvent $event): void
    {
        $transaction = $event->getTransaction();

        if ($event->getOutputCode() === RequestProcessor::OUTPUT_CODE_FAILED && !$transaction->getDataCell('cart_items')) {
            // Failed transaction: Register info about cart items
            $event->addOutputTransactionData(
                'cart_items',
                serialize($transaction->getCartItems()),
                'Cart items'
            );
        }

        $description = $this->translation->translate($transaction->getHistoryEventDescription(), $transaction->getHistoryEventDescriptionData());

        $this->orderHistory->registerTransaction(
            $transaction->getOrder()->getOrderId(),
            $description,
            $transaction->getEventData()
        );

        if ($event->getOutputCode() === RequestProcessor::OUTPUT_CODE_FAILED) {
            // Send notification 'Failed transaction' to the Orders department
            Mailer::sendFailedTransactionAdmin($transaction);
        }
    }
}
