<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Event\Payment;

use XLite\Model\Payment\Method;

trait PaymentActionEventTrait
{
    public function getData(): array
    {
        if ($this->inputData === null) {
            $this->inputData = [];
            foreach ($this->input->getActionData() ?? [] as $item) {
                $this->inputData[$item->getName()] = $item->getValue();
            }
        }

        return $this->inputData;
    }

    public function getMetaData(): array
    {
        if ($this->inputMetaData === null) {
            $this->inputMetaData = [];
            foreach ($this->input->getActionMetaData() ?? [] as $item) {
                $this->inputMetaData[$item->getName()] = $item->getValue();
            }
        }

        return $this->inputMetaData;
    }

    public function addOutputTransactionData(string $name, string $value, ?string $label = null, ?string $accessLevel = null): self
    {
        $this->outputTransactionData[] = $this->transformTransactionData([$name, $value, $label, $accessLevel]);

        return $this;
    }

    public function addOutputBackendTransaction(string $status, float $value, string $type, array $data = []): self
    {
        $this->outputBackendTransaction = (new BackendTransaction())
            ->setStatus($status)
            ->setValue($value)
            ->setType($type)
            ->setData(array_map(fn ($data) => $this->transformTransactionData($data), $data));

        return $this;
    }

    public function getAction(): string
    {
        return $this->input->getAction();
    }

    public function getMethodServiceName(): string
    {
        return $this->transaction?->getPaymentMethod()->getAdaptedServiceName() ?: 'offline';
    }

    public function checkTotal(float $total): bool
    {
        $currency = $this->transaction->getCurrency();

        if ($total && $currency->roundValue($this->transaction->getValue()) !== $currency->roundValue($total)) {
            $this->addOutputTransactionData(
                'total_checking_error',
                sprintf('Total amount doesn\'t match. Transaction total: %s; payment gateway amount: %s', $this->transaction->getValue(), $total),
                'Hacking attempt'
            );

            return false;
        }

        return true;
    }

    public function getProcessorServiceName(): string
    {
        $class = $this->transaction?->getPaymentMethod()->getClass();

        $class = str_replace('\\', '.', $class);
        $class = preg_replace('/([a-z]{2})([A-Z])/', '$1_$2', lcfirst((string) $class));

        return strtolower($class);
    }

    public function getMethod(): ?Method
    {
        return $this->transaction?->getPaymentMethod();
    }

    private function transformTransactionData($data): TransactionData
    {
        [$name, $value, $label, $accessLevel] = $data;

        return (new TransactionData())
            ->setName($name)
            ->setValue($value)
            ->setLabel($label)
            ->setAccessLevel($accessLevel);
    }
}
