<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Command\Service;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use XLite;
use Symfony\Component\Yaml\Yaml;
use XLite\Core\Translation;
use XLite\Model\Repo\ModelRepoTranslationTrait;

final class GetLangPackCommand extends Command
{
    use CommandHelpersTrait;

    protected static $defaultName = 'xcart:service:get-lang-pack';

    protected function getOptionDescriptions(): array
    {
        return [
            'to'   => 'Path to the CSV file to write the language dump, relative to X-Cart root path. If the file exists, it will be overwritten.',
            'lang' => [
                'text'    => 'A two-letter code of the language to dump.',
                'default' => 'en'
            ]
        ];
    }

    protected function getCmdDescription(): string
    {
        return 'Dumps all translatable language labels to a CSV file.';
    }

    protected function configure()
    {
        $this
            ->setDescription($this->getCmdDescription())
            ->setHelp(
                $this->generateHelp(
                    [
                        'options' => [
                            '--to',
                            ['--lang', '-l']
                        ],
                        'examples' => [
                            '--to=lang_dump.csv', '--lang=es --to=lang/es.csv', '-lde --to=lang/de.csv'
                        ]
                    ]
                )
            )
            ->addOption(
                'to',
                null,
                InputArgument::REQUIRED,
                $this->getOptionDescription('to'),
                $this->getOptionDefaultValue('to')
            )
            ->addOption(
                'lang',
                'l',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('lang'),
                $this->getOptionDefaultValue('lang')
            )
            ->addFromServiceToolOption();
    }

    protected function getTranslationsToExportFromYamlFiles(string $lang, array $translations = []): array
    {
        $yamlFiles = array_merge(
            glob('sql/*.yaml', GLOB_NOSORT),
            glob('sql/*/*.yaml', GLOB_NOSORT),
            glob('modules/*/*/config/install.yaml', GLOB_NOSORT)
        );
        $result = $translations;
        $translationsToExport = Translation::getModelsToExportTranslations();

        foreach ($yamlFiles as $file) {
            $data = Yaml::parseFile($file);

            if ($data) {
                foreach ($translationsToExport as $modelFQN) {
                    if (isset($data[$modelFQN])) {
                        /** @var ModelRepoTranslationTrait $repo */
                        $repo = XLite\Core\Database::getRepo($modelFQN);

                        if (
                            $repo
                            && method_exists($repo, 'getTranslationDataFromYamlStructure')
                        ) {
                            foreach ($data[$modelFQN] as $entity) {
                                if ($tData = $repo->getTranslationDataFromYamlStructure($entity, $lang)) {
                                    $result[$modelFQN][] = $tData;
                                }
                            }
                        }
                    }
                }
            }
        }

        return $result;
    }

    protected function getTranslationsToExportFromDatabase(string $lang, array $translations = []): array
    {
        $result = $translations;

        foreach (Translation::getModelsToExportTranslations() as $modelFQN) {
            if (
                ($repo = XLite\Core\Database::getRepo($modelFQN))
                && (method_exists($repo, 'getTranslationModelFQN'))
                && ($tRepo = XLite\Core\Database::getRepo($repo->getTranslationModelFQN()))
            ) {
                $translations = $tRepo->findBy([
                    'code' => $lang,
                ]);

                /** @var XLite\Model\Base\Translation $translation */
                foreach ($translations as $translation) {
                    if ($tData = $repo->getTranslationDataFromModel($translation)) {
                        $result[$modelFQN][] = $tData;
                    }
                }
            }
        }

        return $result;
    }

    protected function writeTranslationsToCSV(string $filePath, array $translations, string $lang): void
    {
        $fp = fopen($filePath, 'wb');

        foreach ($translations as $modelFQN => $group) {
            foreach ($group as $translationData) {
                fputcsv($fp, array_merge([ $lang, $modelFQN ], $translationData));
            }
        }

        fclose($fp);
    }

    /**
     * @return array<string, string[][]>
     */
    protected function removeTranslationDuplicates(array $translations): array
    {
        $uTranslations = [];

        foreach (Translation::getModelsToExportTranslations() as $modelFQN) {
            if (!empty($translations[$modelFQN])) {
                $uGroup = [];

                /** @var ModelRepoTranslationTrait $repo */
                if ($repo = XLite\Core\Database::getRepo($modelFQN)) {
                    $searchParamCount = $repo->getTranslationSearchCellsCount();

                    foreach ($translations[$modelFQN] as $translation) {
                        $key = implode(
                            '-',
                            array_slice($translation, 0, $searchParamCount)
                        );

                        $uGroup[$key] = $translation;
                    }
                }

                $uTranslations[$modelFQN] = array_values($uGroup);
            }
        }

        return $uTranslations;
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $lang = $input->getOption('lang') ?: XLite::getDefaultLanguage();
        $outputFilePath = LC_DIR_ROOT . ($input->getOption('to') ?: "labels-{$lang}-" . date('Y-m-d-H.i.s') . '.csv');

        $this->setInterfaces($input, $output);

        $this->writeTranslationsToCSV(
            $outputFilePath,
            $this->removeTranslationDuplicates(
                $this->getTranslationsToExportFromDatabase(
                    $lang,
                    $this->getTranslationsToExportFromYamlFiles($lang)
                )
            ),
            $lang
        );

        $this->writeln("Language to export: <info>{$lang}</info>");
        $this->writeln("Labels exported to <info>{$outputFilePath}</info>");

        return $this->returnResult(Command::SUCCESS);
    }
}
