<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Command\Service;

use JsonException;
use Psr\Log\LoggerInterface;
use SebastianBergmann\Template\RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use XCart\Operation\Service\Translation\TranslationCreateBase;
use XCart\Operation\Service\Translation\TranslationCreateBaseConfig;
use XCart\Operation\Service\Translation\TranslationParser;
use XLite\Core\Translation;

final class GenerateTranslationBaseCommand extends Command
{
    use CommandHelpersTrait;

    protected const MODULE_TEMP_DIR = LC_DIR_ROOT . 'var/packs/xcart';

    /**
     * @var string
     */
    protected static $defaultName = 'xcart:service:generate-translation-base';

    protected function getCmdDescription(): string
    {
        return 'Generate install.yaml for the translation module from CSV file.';
    }

    protected function getArgumentDescriptions(): array
    {
        return [
            'src' => 'CSV file path (relative to the X-Cart root folder).'
        ];
    }

    protected function getOptionDescriptions(): array
    {
        return [
            'lang'   => [
                'text'    => 'Language code.',
                'default' => 'language code from the first line in the CSV file',
            ],
            'author' => [
                'text'    => 'Module author',
                'default' => 'XC',
            ],
            'name'   => [
                'text'    => 'Module name',
                'default' => '"{language code}Translation"',
            ],
        ];
    }

    public function __construct(
        protected LoggerInterface $logger,
        protected TranslationParser $parser,
        protected TranslationCreateBase $createBase
    ) {
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDescription($this->getCmdDescription())
            ->setHelp(
                $this->generateHelp([
                    'arguments' => [ 'src' ],
                    'options'   => [
                        [ '--lang', '-l' ],
                        [ '--author', '-a' ],
                        '--name'
                    ],
                    'examples'  => [
                        'es.csv',
                        '--lang=es es-labels.csv',
                        '--author=XC --name=EsTranslation --lang=es es.csv',
                        '-lde de.csv'
                    ]
                ])
            )
            ->addOption(
                'lang',
                'l',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('lang')
            )
            ->addOption(
                'author',
                'a',
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('author'),
                $this->getOptionDefaultValue('author')
            )
            ->addOption(
                'name',
                null,
                InputOption::VALUE_REQUIRED,
                $this->getOptionDescription('name')
            )
            ->addArgument(
                'src',
                InputArgument::REQUIRED,
                $this->getOptionDescription('src')
            )
            ->addFromServiceToolOption();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->setInterfaces($input, $output);

        $src = LC_DIR_ROOT . $input->getArgument('src');

        if (!file_exists($src)) {
            return $this->returnError("CSV file {$src} not found.");
        }

        if (!is_readable($src)) {
            return $this->returnError("Cannot read CSV file {$src}.");
        }

        $parsed = ($this->parser)(
            $src,
            Translation::getModelsToIncludeIntoLanguageModule(),
            (string)($input->getOption('lang') ?: '')
        );

        if (!($lang = $parsed->getLngCode())) {
            return $this->returnError("Cannot detect language code.", [
                'CSV file' => $src
            ]);
        }

        if ($parsed->getYamlEntitiesCount() === 0) {
            return $this->returnError('No labels can be added.', [
                'CSV file' => $src
            ]);
        }

        $moduleId = $input->getOption('name')
            ?: strtoupper(($lang[0] ?? '')) . strtolower(substr($lang, 1)) . 'Translation';

        try {
            $createBaseResult = ($this->createBase)(
                (new TranslationCreateBaseConfig())
                    ->setAuthorId($input->getOption('author'))
                    ->setData($parsed)
                    ->setModuleNameId((string)$input->getOption('name'))
            );
        } catch (RuntimeException $e) {
            return $this->returnError(
                "Error while writing translation module base: {$e->getMessage()}.",
                [
                    'CSV file' => $src,
                    'Language code' => $lang
                ]
            );
        }

        if ($this->isCalledFromServiceTool()) {
            try {
                $translationBaseResult = [
                    'author'      => $input->getOption('author'),
                    'moduleName'  => $moduleId,
                    'lang'        => $lang,
                    'entities'    => $parsed->getYamlEntitiesCount(),
                    'ignored'     => $parsed->getIgnoredEntitiesCount(),
                    'dir'         => $createBaseResult->getModuleDir(),
                    'ignoredPath' => $createBaseResult->getIgnoredLabelsFilePath()
                ];

                $this->writeln(
                    json_encode($translationBaseResult, JSON_THROW_ON_ERROR)
                );

                $this->logger->debug("Translation base successfully generated.", $translationBaseResult);
            } catch (JsonException $e) {
                return $this->returnError('Error during the answer JSON generation: ' . $e->getMessage());
            }
        } else {
            $this->writeln("Module language: <info>{$lang}</info>");
            $this->writeln("<options=bold>{$parsed->getYamlEntitiesCount()}</> labels have been added, module base path: <info>{$createBaseResult->getModuleDir()}</info>");

            if (($ignored = $parsed->getIgnoredEntitiesCount()) > 0) {
                $this->writeln("<options=bold>{$ignored}</> labels ignored, full list is in <info>{$createBaseResult->getIgnoredLabelsFilePath()}</info>");
            }
        }

        return $this->returnResult(Command::SUCCESS);
    }
}
