<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Command\Service;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

trait CommandHelpersTrait
{
    protected string $outText = '';

    protected ?InputInterface $input = null;

    protected ?OutputInterface $output = null;

    protected function getOptionDescriptions(): array
    {
        return [];
    }

    protected function getArgumentDescriptions(): array
    {
        return [];
    }

    protected function getCmdDescription(): string
    {
        return '';
    }

    /**
     * @return array{
     *     sectionTitle: string,
     *     option: string,
     *     optionDefaultValue: string,
     *     example: string
     * }
     */
    protected function getStyles(): array
    {
        return [
            'sectionTitle'       => '<fg=yellow>',
            'option'             => '<fg=green>',
            'optionDefaultValue' => '<fg=yellow>',
            'example'            => '<fg=green>'
        ];
    }

    protected function getOptionColumnLength(string|array $option): int
    {
        return strlen(
            implode(
                ', ',
                (array)$option
            )
        );
    }

    protected function getMaxOptionsColumnLength(array $options): int
    {
        return array_reduce(
            $options,
            fn (int $carry, string|array $option): int => max(
                $carry,
                $this->getOptionColumnLength($option)
            ),
            0
        );
    }

    protected function searchInDescriptionArrays(array|string $option)
    {
        if (is_string($option)) {
            $optionDescriptions = $this->getOptionDescriptions();
            $argumentDescriptions = $this->getArgumentDescriptions();

            if (isset($optionDescriptions[$option])) {
                $option = $optionDescriptions[$option];
            } elseif (isset($argumentDescriptions[$option])) {
                $option = $argumentDescriptions[$option];
            }
        }

        return $option;
    }

    protected function getOptionDescription(array|string $description, bool $printDefaultValue = false): string
    {
        $result = '';

        $description = $this->searchInDescriptionArrays($description);

        if (!is_array($description)) {
            $description = ['text' => $description];
        }

        if (!empty($description['text'])) {
            $result .= $description['text'];
        }

        if ($printDefaultValue && !empty($description['default'])) {
            $result .= (empty($result) ? '' : ' ') . $this->getStyles()['optionDefaultValue'] . '[default: '
                       . $description['default'] . ']</>';
        }

        return $result;
    }

    protected function getOptionDefaultValue(array|string $option): mixed
    {
        return ((array)$this->searchInDescriptionArrays($option))['default'] ?? null;
    }

    protected function getSectionTitle(string $title): string
    {
        return "\n" . $this->getStyles()['sectionTitle'] . "{$title}:</>\n";
    }

    protected function generateOptionsSection(string $title, array $options, array $descriptionsArray): string
    {
        $result = $this->getSectionTitle($title);
        $optionsColumnLength = $this->getMaxOptionsColumnLength($options);
        $optionStyle = $this->getStyles()['option'];

        foreach ($options as $option) {
            $variants = (array)($option ?? '');
            $optionLength = $this->getOptionColumnLength($option);
            $description = '';
            $givenDescription = $descriptionsArray[trim($variants[0], " \t\n\r\0\x0B-")] ?? '';

            if (!empty($givenDescription)) {
                $description = is_array($givenDescription)
                    ? $givenDescription
                    : [ 'text' => $givenDescription ];
            }

            $result .= '  ' . $optionStyle
                       . implode(
                           '</>, ' . $optionStyle,
                           $variants
                       )
                       . '</>'
                       . str_repeat(' ', max(0, ($optionsColumnLength - $optionLength)))
                       . '  ' . $this->getOptionDescription($description, true) . "\n";
        }

        return $result;
    }

    protected function generateExamplesSection(array $examples): string
    {
        $result = '';

        if (!empty($examples)) {
            $result = $this->getSectionTitle('For example');

            foreach ($examples as $example) {
                $result .= '  ' . $this->getStyles()['example'] . './bin/console ' . self::$defaultName . ' ' . $example . "</>\n";
            }
        }

        return $result;
    }

    protected function generateHelp(array $sections): string
    {
        $result = "\n" . $this->getCmdDescription() . "\n";

        if (!empty($sections['arguments'])) {
            $result .= $this->generateOptionsSection('Arguments', $sections['arguments'], $this->getArgumentDescriptions());
        }

        if (!empty($sections['options'])) {
            $result .= $this->generateOptionsSection('Options', $sections['options'], $this->getOptionDescriptions());
        }

        if (!empty($sections['examples'])) {
            $result .= $this->generateExamplesSection($sections['examples']);
        }

        return $result;
    }

    protected function setInterfaces(InputInterface $input, OutputInterface $output): void
    {
        $this->input = $input;
        $this->output = $output;
    }

    protected function isCalledFromServiceTool(): bool
    {
        return $this->input && $this->input->getOption('from-service-tool');
    }

    protected function write(string $text): void
    {
        if ($this->output && $this->isCalledFromServiceTool()) {
            $this->outText .= $text;
        } else {
            $this->output->write($text);
        }
    }

    protected function writeln(string $line): void
    {
        if ($this->output && $this->isCalledFromServiceTool()) {
            $this->outText .= "{$line}\n";
        } else {
            $this->output->writeln($line);
        }
    }

    protected function addFromServiceToolOption(): void
    {
        $this->addOption(
            'from-service-tool',
            null,
            InputOption::VALUE_NONE,
            'Service option to detect if the command is run from service tool.'
        );
    }

    protected function returnResult(int $result): int
    {
        if ($this->isCalledFromServiceTool() && $this->output) {
            $this->output->write(
                json_encode([
                    'result' => match ($result) {
                        Command::FAILURE => 'FAILURE',
                        Command::INVALID => 'INVALID',
                        default          => 'SUCCESS'
                    },
                    'output' => $this->outText
                ]),
                false,
                $this->output::OUTPUT_RAW
            );

            return Command::SUCCESS;
        }

        return $result;
    }

    protected function returnError(string $errorText, array $context = [], int $errorCode = Command::FAILURE): int
    {
        $this->writeln("<error>$errorText</error>");

        $this->logger->error($errorText, $context);

        return $errorCode;
    }
}
