<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\Command\GenerateData;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use XCart\Event\EntityGenerator\OrderPreClearEvent;
use XLite\Console\Command\GenerateData\Generators\Category;
use XLite\Console\Command\GenerateData\Generators\Image;
use XLite\Console\Command\GenerateData\Generators\Order;
use XLite\Console\Command\GenerateData\Generators\Product;
use XLite\Console\Command\GenerateData\GlobalRuntimeCache;
use XLite\Core\Database;
use XLite\Model\Attribute;
use XLite\Model\AttributeOption;

/**
 * Class StoreDataToYamlCommand
 * @package XCart\Command\GenerateData
 */
class GenerateDataCommand extends Command
{
    public function __construct(
        private EventDispatcherInterface $eventDispatcher
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName('generate:all')
            ->setDescription('Generate entities')
            ->setHelp('Generates entities of various types up to the specified limit. Removes all existing entities of selected type before the process.')

            ->addOption('categories', 'c', InputOption::VALUE_REQUIRED, 'Categories count per level')
            ->addOption('categoryImage', 'I', InputOption::VALUE_NONE, 'Generate categories with image')
            ->addOption('depth', 'd', InputOption::VALUE_REQUIRED, 'Categories depth')

            ->addOption('products', 'p', InputOption::VALUE_REQUIRED, 'Product per category')
            ->addOption('globalAttributes', 'g', InputOption::VALUE_REQUIRED, 'Global attributes per product', 2)
            ->addOption('globalAttributeOptions', 'G', InputOption::VALUE_REQUIRED, 'Options per global attribute', 3)
            ->addOption('options', 'o', InputOption::VALUE_REQUIRED, 'Product-specific attributes per product')
            ->addOption('optionsValues', 'val', InputOption::VALUE_REQUIRED, 'Option values per product-specific attribute')
            ->addOption('productImages', 'i', InputOption::VALUE_REQUIRED, 'Images per product')
            ->addOption('wholesalePrices', 'w', InputOption::VALUE_REQUIRED, 'Wholesale prices per product')

            ->addOption('imagesType', 't', InputOption::VALUE_REQUIRED, 'Images type: "same"(same file), "unique"(autogenerated content)')

            ->addOption('featuredProducts', 'f', InputOption::VALUE_REQUIRED, 'Featured products per category')

            ->addOption('orders', 'r', InputOption::VALUE_REQUIRED, 'Orders to create')
            ->addOption('orderItems', 'O', InputOption::VALUE_REQUIRED, 'Orders items to create')
            ->addOption('ordersPeriod', 'P', InputOption::VALUE_REQUIRED, 'Orders date period in days')
            ->addOption('silent', 's', InputOption::VALUE_NONE, 'Suppress all confirmation (unconfirmed by default)')
        ;
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $io->title('Generating data');

        $cats = (int) ($input->getOption('categories') ?: 0);
        $depth = (int) ($input->getOption('depth') ?: 1);
        $counts = [
            'categories'                 => $cats,
            'depth'                      => $depth,
            'categoriesToBeGenerated'    => $cats === 1 ? $depth : pow($cats, $depth),
            'featuredProducts'           => $input->getOption('featuredProducts') ?: 0,
            'products'                   => $input->getOption('products') ?: 0,
            'globalAttributes'           => $input->getOption('globalAttributes') ?: 0,
            'globalAttributeOptions'     => $input->getOption('globalAttributeOptions') ?: 0,
            'options'                    => $input->getOption('options') ?: 0,
            'optionsValues'              => $input->getOption('optionsValues') ?: 0,
            'productImages'              => $input->getOption('productImages') ?: 0,
            'wholesalePrices'            => $input->getOption('wholesalePrices') ?: 0,
            'orders'                     => $input->getOption('orders') ?: 0,
            'orderItems'                 => $input->getOption('orderItems') ?: 0,
            'ordersPeriod'               => $input->getOption('ordersPeriod') ?: 0,
            'categoryImage'              => $input->getOption('categoryImage') ?: 0,
        ];

        $io->section('Entities to generate (count)');
        $io->table([ 'name', 'value' ], array_map(null, array_keys($counts), array_values($counts)));

        if ($counts['categories']) {
            $io->section('Generating categories');
            $this->generateCategories($io, $input, $counts);
            $io->success('Categories generated');
        }

        if ($counts['globalAttributes']) {
            $io->section('Generating global attributes');
            $this->generateGlobalAttributes($io, $counts);
            $io->success('Global attributes generated');
        }

        if ($counts['products']) {
            $io->section('Generating products');
            $this->generateProducts($io, $input, $counts);
            $io->success('Products generated');
        }

        if ($counts['orders']) {
            $io->section('Generating orders');
            $this->generateOrders($io, $input, $counts);
            $io->success('Orders generated');
        }

        $io->success('Finished');
        Image::clearTmpDir();

        return Command::SUCCESS;
    }

    protected function generateCategories(SymfonyStyle $io, InputInterface $input, $counts)
    {
        $repo = Database::getRepo('XLite\Model\Category');

        if (!$input->getOption('silent') && $io->confirm('Clear categories?', false)) {
            $io->write('Removing all existing categories');

            $this->clearCategories();

            $io->writeln('[ OK ]');
        }

        $io->newLine();
        $io->write('<info>Generating categories</info>');

        $this->doGenerateCategories(
            $io,
            '0',
            $counts['categories'],
            $counts['depth'],
            $counts['categoryImage'],
            new Image($input->getOption('imagesType'))
        );
        $io->newLine();
        $io->writeln('[ Generated ]');

        Database::getEM()->flush();
        Database::getEM()->clear();

        $io->newLine();
        $io->write('<info>Recalculating quick flags</info>');
        $repo->correctCategoriesStructure();
        $io->writeln('[ OK ]');

        Database::getEM()->clear();
    }

    protected function doGenerateCategories(SymfonyStyle $io, $suffix, $count, $maxDepth, $generateImage, $imageGenerator, \XLite\Model\Category $parent = null, $depth = 1)
    {
        if (!$parent) {
            $parent = Database::getRepo('XLite\Model\Category')->getRootCategory();
        }

        $commonProgress = $io->createProgressBar($count);

        $commonProgress->start();
        if ($depth === 1) {
            $io->newLine();
        }

        for ($i = 0; $i < $count; $i++) {
            $generator = new Category();
            $category = $generator->generate(
                $suffix . '_' . $i,
                $i,
                $parent,
                $generateImage,
                $imageGenerator
            );

            if ($depth === 1) {
                $io->write("\033[1A");
            }
            $commonProgress->advance();

            if ($depth === 1) {
                $io->newLine();
            }

            if ($depth < $maxDepth) {
                $this->doGenerateCategories($io, $suffix . '_' . $i, $count, $maxDepth, $generateImage, $imageGenerator, $category, $depth + 1);
            }
        }

        Database::getEM()->flush();
        Database::getEM()->clear();

        $commonProgress->finish();
        $commonProgress->clear();
    }

    protected function generateGlobalAttributes(SymfonyStyle $io, array $counts): void
    {
        $attributeRepo = Database::getRepo(Attribute::class);
        $attributeOptionRepo = Database::getRepo(AttributeOption::class);

        $commonProgress = $io->createProgressBar($counts['globalAttributes']);
        $commonProgress->start();

        for ($i = 1; $i <= $counts['globalAttributes']; $i++) {
            $attribute = $attributeRepo->insert(
                [
                    'name' => "Test attribute $i",
                ],
                false
            );
            Database::getEM()->persist($attribute);
            GlobalRuntimeCache::$attributesCache[] = $attribute;

            for ($j = 1; $j <= $counts['globalAttributeOptions']; $j++) {
                $option = $attributeOptionRepo->insert(
                    [
                        'attribute' => $attribute,
                        'name'      => "Value $i-$j",
                        'addToNew'  => true,
                    ],
                    false
                );
                Database::getEM()->persist($option);
            }
            $commonProgress->advance();
        }

        Database::getEM()->flush();
        Database::getEM()->clear();
    }

    protected function generateProducts(SymfonyStyle $io, InputInterface $input, $counts)
    {
        if (!$input->getOption('silent') && $io->confirm('Remove all existing products?', false)) {
            $io->write('Removing all existing products');
            $this->clearProducts();
            $io->writeln('[ OK ]');
        }

        $categoriesCount = Database::getRepo('XLite\Model\Category')->count();
        $categories = Database::getRepo('XLite\Model\Category')->iterateAll();

        $io->newLine();
        $io->writeln('<info>Generating products ... </info>');
        $commonProgress = $io->createProgressBar(($categoriesCount - 1) * $counts['products']);

        $generator = new Product(
            $counts['options'],
            $counts['optionsValues'],
            $counts['productImages'],
            $counts['wholesalePrices'],
            new Image($input->getOption('imagesType'))
        );

        $productsGenerated = 0;
        $batchSize = 100;

        /** @var \XLite\Model\Category $category */
        foreach ($categories as $category) {
            $featuredInCategory = 0;

            for ($i = 0; $i < $counts['products']; $i++) {
                $product = $generator->generate($category, 'P' . $i);

                if (class_exists('CDev\FeaturedProducts\Model\FeaturedProduct')) {
                    if ($featuredInCategory < $counts['featuredProducts']) {
                        $this->createFeaturedProduct($category, $product, $i);

                        $featuredInCategory++;
                    }
                }

                $commonProgress->advance();

                if ($productsGenerated % $batchSize === 0) {
                    Database::getEM()->flush();
                    Database::getEM()->clear();
                    $category = Database::getEM()->merge($category);
                }
                $productsGenerated++;
            }
        }

        $commonProgress->finish();
        $io->newLine(2);

        Database::getEM()->flush();
        Database::getEM()->clear();

        $io->newLine();
        $io->writeln('<info>Recalculating quick data ... </info>');
        $this->recalculateProductsQuickData($io);
        $io->writeln('[ OK ]');
    }

    protected function clearProducts()
    {
        Database::getRepo('XLite\Model\Product')->createPureQueryBuilder()
            ->delete('XLite\Model\Product', 'p')
            ->execute();

        Database::getEM()->flush();
        Database::getEM()->clear();
    }

    protected function clearCategories()
    {
        Database::getRepo('XLite\Model\Category')->createPureQueryBuilder()
            ->delete('XLite\Model\Category', 'c')
            ->where('c.category_id != :rootCategory')
            ->setParameter('rootCategory', Database::getRepo('XLite\Model\Category')->getRootCategoryId())
            ->execute();

        Database::getEM()->flush();
        Database::getEM()->clear();
    }

    protected function recalculateProductsQuickData(SymfonyStyle $io)
    {
        $quickData = \XLite\Core\QuickData::getInstance();
        $progress = $io->createProgressBar($quickData->countUnprocessed());
        $progress->display();

        do {
            $processed = $quickData->updateUnprocessedChunk(\XLite\Core\QuickData::CHUNK_LENGTH);
            if (0 < $processed) {
                Database::getEM()->clear();
            }
            $progress->advance($processed);
        } while (0 < $processed);
        $progress->finish();
    }

    /**
     * @param $category
     * @param $product
     * @param $i
     *
     * @return \CDev\FeaturedProducts\Model\FeaturedProduct
     */
    protected function createFeaturedProduct($category, $product, $i)
    {
        return Database::getRepo('CDev\FeaturedProducts\Model\FeaturedProduct')->insert(
            [
                'product'  => $product,
                'category' => $category,
                'orderBy'  => $i,
            ],
            false
        );
    }

    protected function generateOrders(SymfonyStyle $io, InputInterface $input, $counts)
    {
        if (!$input->getOption('silent') && $io->confirm('Remove all existing orders?', false)) {
            $io->write('Removing all existing orders');
            $this->clearOrders();
            $io->writeln('[ OK ]');
        }

        /** @var \XLite\Model\Profile $profile */
        $profile = Database::getRepo('XLite\Model\Profile')->createQueryBuilder()
            ->bindRegistered()
            ->getSingleResult();

        /** @var \XLite\Model\Currency $currency */
        $currency = Database::getRepo('XLite\Model\Currency')->findOneByCode('USD');
        /** @var \XLite\Model\Payment\Method $method */
        $method = Database::getRepo('XLite\Model\Payment\Method')->findByServiceName('PhoneOrdering');

        $generator = new Order();
        $count = $counts['orders'];
        $itemsCount = $counts['orderItems'];
        $period = $counts['ordersPeriod'];
        $batchSize = 5;

        $commonProgress = $io->createProgressBar($count);

        for ($i = 0; $i < $count; $i++) {
            if (!Database::getEM()->contains($profile)) {
                $profile = Database::getRepo('XLite\Model\Profile')->find($profile->getProfileId());
                $currency = Database::getRepo('XLite\Model\Currency')->find($currency->getCurrencyId());
            }

            $generator->generate($profile, $currency, $method, $itemsCount, $period);

            if ($i % $batchSize == 0) {
                Database::getEM()->flush();
                Database::getEM()->clear();
            }
            $commonProgress->advance();
        }
        $commonProgress->finish();
        $io->newLine(2);
        Database::getEM()->flush();

        $io->write('Updating sales...');
        $this->updateSales();
        $io->write('[ OK ]');

        Database::getEM()->clear();
    }

    protected function clearOrders()
    {
        $this->eventDispatcher->dispatch(new OrderPreClearEvent(), OrderPreClearEvent::NAME);

        Database::getRepo('XLite\Model\Order')->createPureQueryBuilder()
            ->delete('XLite\Model\Order', 'o')
            ->execute();

        Database::getEM()->flush();
        Database::getEM()->clear();
    }

    protected function updateSales()
    {
        $em = Database::getEM();
        $batchSize = 20;
        $i = 0;
        $q = $em->createQuery('select p from XLite\Model\Product p');
        $iterableResult = $q->iterate();
        foreach ($iterableResult as $row) {
            /** @var \Xlite\Model\Product $product */
            $product = $row[0];
            $product->updateSales();
            if (($i % $batchSize) === 0) {
                $em->flush(); // Executes all updates.
                $em->clear(); // Detaches all objects from Doctrine!
            }
            ++$i;
        }
        $em->flush();
    }
}
