<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\ApiPlatform\Core\OpenApi\Factory;

use ApiPlatform\Core\OpenApi\Factory\OpenApiFactoryInterface;
use ApiPlatform\Core\OpenApi\Model\Paths;
use ApiPlatform\Core\OpenApi\OpenApi;
use Symfony\Component\HttpFoundation\RequestStack;

class SwaggerUIOpenApiFactory implements OpenApiFactoryInterface
{
    /**
     * @param array<string,string[]> $sectionHashMap
     */
    public function __construct(
        private OpenApiFactoryInterface $inner,
        private RequestStack $requestStack,
        private array $sectionHashMap,
    ) {
    }

    public function __invoke(array $context = []): OpenApi
    {
        $request = $this->requestStack->getMainRequest();
        if (!$request) {
            return $this->inner->__invoke($context);
        }

        $section = $request->get('section');
        if (empty($section)) {
            $section = key($this->sectionHashMap);
        }

        $result = $this->inner->__invoke($context);

        $paths = new Paths();
        foreach ($result->getPaths()->getPaths() as $path => $pathItem) {
            if ($this->isAllowed($path, $section)) {
                $paths->addPath($path, $pathItem);
            }
        }

        if ($section === 'storefront') {
            $description = <<<TXT
X-Cart Storefront API

        # Authentication

        All REST API queries require a valid JWT Authorization token.<br /><br />
        You can obtain JWT token on /api/storefront/auth/login endpoint.<br /><br />
        Be sure to include your token as an **Authorization:** header (like "**Authorization:** Bearer <JWT token>") on all API queries.

        # Localization

        X-Cart REST API supports multiple languages, but all the requests are made in the same language. The language is set using the **Accept-Language** request HTTP header
        [https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Accept-Language](https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Accept-Language)<br /><br />
        If the header has not been sent, the default language as specified in the store settings is used.
TXT;
            $result = $result->withInfo($result->getInfo()->withDescription($description));
            $result = $result->withInfo($result->getInfo()->withTitle('X-Cart Storefront API'));
        }

        return $result->withPaths($paths);
    }

    private function isAllowed(string $path, string $section): bool
    {
        $prefixes = $this->sectionHashMap[$section];
        if (!empty($prefixes)) {
            return $this->isPathStartsWithPrefixes($path, $prefixes);
        }

        foreach ($this->sectionHashMap as $key => $prefixes) {
            if ($key !== $section && $this->isPathStartsWithPrefixes($path, $prefixes)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param string[]  $prefixes
     */
    private function isPathStartsWithPrefixes(string $path, array $prefixes): bool
    {
        foreach ($prefixes as $prefix) {
            if (str_starts_with($path, $prefix)) {
                return true;
            }
        }

        return false;
    }
}
