<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\API;

use ApiPlatform\Core\OpenApi\Factory\OpenApiFactoryInterface;
use ApiPlatform\OpenApi\Model\MediaType;
use ApiPlatform\OpenApi\Model\Response;
use ApiPlatform\Core\OpenApi\OpenApi;
use ApiPlatform\OpenApi\Model\PathItem;
use ArrayObject;

class OpenApiFactorySchemasNormalizationDecorator implements OpenApiFactoryInterface
{
    public function __construct(
        private OpenApiFactoryInterface $inner,
    ) {
    }

    public function __invoke(array $context = []): OpenApi
    {
        $api = $this->inner->__invoke($context);

        $usedSchemas = $this->getUsedSchemas($api);
        do {
            [$usedSchemas, $count] = $this->enrichSubSchemas($api, $usedSchemas);
        } while ($count > 0);

        return $this->filterSchemas($api, $usedSchemas);
    }

    /**
     * @return string[]
     */
    private function getUsedSchemas(OpenApi $api): array
    {
        $usedModels = [];

        /** @var PathItem $pathItem */
        foreach ($api->getPaths()->getPaths() as $pathItem) {
            if ($pathItem->getGet()) {
                $usedModels[] = $this->getUsedSchemasFromResponses($pathItem->getGet()->getResponses());
            }

            if ($pathItem->getPost()) {
                $usedModels[] = $this->getUsedSchemasFromResponses($pathItem->getPost()->getResponses());

                if ($pathItem->getPost()->getRequestBody()) {
                    $usedModels[] = $this->getUsedSchemasFromRequests($pathItem->getPost()->getRequestBody()->getContent());
                }
            }

            if ($pathItem->getPut()) {
                $usedModels[] = $this->getUsedSchemasFromResponses($pathItem->getPut()->getResponses());

                if ($pathItem->getPut()->getRequestBody()) {
                    $usedModels[] = $this->getUsedSchemasFromRequests($pathItem->getPut()->getRequestBody()->getContent());
                }
            }

            if ($pathItem->getPatch()) {
                $usedModels[] = $this->getUsedSchemasFromResponses($pathItem->getPatch()->getResponses());

                if ($pathItem->getPatch()->getRequestBody()) {
                    $usedModels[] = $this->getUsedSchemasFromRequests($pathItem->getPatch()->getRequestBody()->getContent());
                }
            }
        }

        $usedModels = array_merge(...$usedModels);

        foreach ($usedModels as $k => $v) {
            $usedModels[$k] = str_replace('#/components/schemas/', '', $v);
        }

        return array_unique($usedModels);
    }

    /**
     * @param Response[] $responses
     *
     * @return string[]
     */
    private function getUsedSchemasFromResponses(array $responses): array
    {
        $usedModels = [];

        foreach ($responses as $response) {
            /** @var MediaType $mediaType */
            foreach ($response->getContent() ?? [] as $mediaType) {
                $schema = $mediaType->getSchema();
                if (!$schema) {
                    continue;
                }

                if (
                    isset($schema['type'], $schema['items'])
                    && $schema['type'] === 'array'
                    && isset($schema['items']['$ref'])
                    && str_starts_with($schema['items']['$ref'], '#/components/schemas/')
                ) {
                    $usedModels[] = $schema['items']['$ref'];
                    continue;
                }

                if (
                    isset($schema['type'])
                    && $schema['type'] === 'object'
                    && isset($schema['properties'], $schema['properties']['hydra:member'])
                    && isset($schema['properties']['hydra:member']['type'])
                    && $schema['properties']['hydra:member']['type'] === 'array'
                    && isset($schema['properties']['hydra:member']['items'])
                    && isset($schema['properties']['hydra:member']['items']['$ref'])
                    && str_starts_with($schema['properties']['hydra:member']['items']['$ref'], '#/components/schemas/')
                ) {
                    $usedModels[] = $schema['properties']['hydra:member']['items']['$ref'];
                    continue;
                }

                if (
                    isset($schema['$ref'])
                    && str_starts_with($schema['$ref'], '#/components/schemas/')
                ) {
                    $usedModels[] = $schema['$ref'];
                }
            }
        }

        return $usedModels;
    }

    private function getUsedSchemasFromRequests(ArrayObject $requests): array
    {
        $usedModels = [];

        /** @var MediaType $mediaType */
        foreach ($requests as $mediaType) {
            $schema = is_object($mediaType)
                ? $mediaType->getSchema()?->getArrayCopy()
                : $mediaType['schema'];
            if (!$schema) {
                continue;
            }

            if (
                isset($schema['$ref'])
                && str_starts_with($schema['$ref'], '#/components/schemas/')
            ) {
                $usedModels[] = $schema['$ref'];
                continue;
            }
        }

        return $usedModels;
    }

    /**
     * @param string[] $usedSchemas
     */
    private function enrichSubSchemas(OpenApi $api, array $usedSchemas): array
    {
        $subSchemas = [];
        $count = count($usedSchemas);
        foreach ($api->getComponents()->getSchemas() as $name => $schema) {
            if (!in_array($name, $usedSchemas, true)) {
                continue;
            }

            $subSchema = $this->findSubSchemas($schema);
            if ($subSchema) {
                $subSchemas[] = $subSchema;
            }
        }

        $usedSchemas = array_merge($usedSchemas, ...$subSchemas);
        $usedSchemas = array_unique($usedSchemas);

        return [$usedSchemas, count($usedSchemas) - $count];
    }

    /**
     * @param string[] $usedSchemas
     */
    private function filterSchemas(OpenApi $api, array $usedSchemas): OpenApi
    {
        $filteredSchemas = new ArrayObject();
        foreach ($api->getComponents()->getSchemas() as $name => $schema) {
            if (!in_array($name, $usedSchemas, true)) {
                continue;
            }

            $filteredSchemas[$name] = $schema;
        }

        return $api->withComponents($api->getComponents()->withSchemas($filteredSchemas));
    }

    private function findSubSchemas($input, $schemas = []): array
    {
        foreach ($input as $k => $v) {
            if ($k === '$ref' && str_starts_with($v, '#/components/schemas/')) {
                $schemas[] = str_replace('#/components/schemas/', '', $v);
                continue;
            }

            if (is_array($v) || $v instanceof ArrayObject) {
                $schemas = $this->findSubSchemas($v, $schemas);
            }
        }

        return $schemas;
    }
}
