<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\API;

use ApiPlatform\Core\OpenApi\Factory\OpenApiFactoryInterface;
use ApiPlatform\OpenApi\Model\MediaType;
use ApiPlatform\OpenApi\Model\Response;
use ApiPlatform\Core\OpenApi\OpenApi;
use ApiPlatform\OpenApi\Model\Operation;
use ApiPlatform\OpenApi\Model\PathItem;
use ArrayObject;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Authorization\Voter\AuthenticatedVoter;
use Symfony\Component\Security\Http\AccessMapInterface;

class OpenApiFactoryResponsesDecorator implements OpenApiFactoryInterface
{
    public function __construct(
        private OpenApiFactoryInterface $inner,
        private AccessMapInterface $map,
    ) {
    }

    public function __invoke(array $context = []): OpenApi
    {
        /** @var OpenApi $api */
        $api = $this->inner->__invoke($context);

        $api->getComponents()->getSchemas()['Error'] = new ArrayObject(
            [
                'type'       => 'object',
                'properties' => [
                    'type'     => [
                        'type'   => 'string',
                        'format' => 'url',
                    ],
                    'title'    => [
                        'type' => 'string',
                    ],
                    'detail'   => [
                        'type'     => 'string',
                        'nullable' => true,
                    ],
                    'instance' => [
                        'type'     => 'string',
                        'nullable' => true,
                    ],
                ],
            ]
        );

        /** @var PathItem $pathItem */
        foreach ($api->getPaths()->getPaths() as $path => $pathItem) {
            // GET
            if ($pathItem->getGet()) {
                $pathItem = $pathItem->withGet($this->addResponse($pathItem->getGet(), 500, 'Internal error'));
                if (!$this->isPublic($path)) {
                    $pathItem = $pathItem->withGet($this->addResponse($pathItem->getGet(), 401, 'Unauthorized'));
                }
                if (!empty($pathItem->getGet()->getResponses()['404'])) {
                    $pathItem = $pathItem->withGet($this->addErrorModel($pathItem->getGet()));
                }
            }

            // POST
            if ($pathItem->getPost()) {
                $pathItem = $pathItem->withPost($this->addResponse($pathItem->getPost(), 500, 'Internal error'));
                if (!$this->isPublic($path)) {
                    $pathItem = $pathItem->withPost($this->addResponse($pathItem->getPost(), 401, 'Unauthorized'));
                }
            }

            // PUT
            if ($pathItem->getPut()) {
                $pathItem = $pathItem->withPut($this->addResponse($pathItem->getPut(), 500, 'Internal error'));
                if (!$this->isPublic($path)) {
                    $pathItem = $pathItem->withPut($this->addResponse($pathItem->getPut(), 401, 'Unauthorized'));
                }
                if (!empty($pathItem->getPut()->getResponses()['404'])) {
                    $pathItem = $pathItem->withPut($this->addErrorModel($pathItem->getPut()));
                }
            }

            // PATCH
            if ($pathItem->getPatch()) {
                $pathItem = $pathItem->withPatch($this->addResponse($pathItem->getPatch(), 500, 'Internal error'));
                if (!$this->isPublic($path)) {
                    $pathItem = $pathItem->withPatch($this->addResponse($pathItem->getPatch(), 401, 'Unauthorized'));
                }
                if (!empty($pathItem->getPatch()->getResponses()['404'])) {
                    $pathItem = $pathItem->withPatch($this->addErrorModel($pathItem->getPatch()));
                }
            }

            // DELETE
            if ($pathItem->getDelete()) {
                $pathItem = $pathItem->withDelete($this->addResponse($pathItem->getDelete(), 500, 'Internal error'));
                if (!$this->isPublic($path)) {
                    $pathItem = $pathItem->withDelete($this->addResponse($pathItem->getDelete(), 401, 'Unauthorized'));
                }
                if (!empty($pathItem->getDelete()->getResponses()['404'])) {
                    $pathItem = $pathItem->withDelete($this->addErrorModel($pathItem->getDelete()));
                }

                if (empty($pathItem->getDelete()->getResponses()['204'])) {
                    $pathItem = $pathItem->withDelete($this->addResponse($pathItem->getDelete(), 204, 'No Content'));
                }
            }

            $api->getPaths()->addPath($path, $pathItem);
        }

        return $api;
    }

    private function isPublic(string $path): bool
    {
        $request = Request::create('http://localhost' . $path);

        return $this->map->getPatterns($request) === [AuthenticatedVoter::PUBLIC_ACCESS];
    }

    private function addResponse(Operation $operation, int $code, string $message): Operation
    {
        $response = new Response(
            $message,
            new ArrayObject(
                [
                    'application/json'         => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'application/ld+json'      => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'application/vnd.api+json' => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'text/html'                => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                ]
            ),
        );

        return $operation->withResponses($operation->getResponses() + [(string)$code => $response]);
    }

    private function addErrorModel(Operation $operation): Operation
    {
        $response = $operation->getResponses()['404']->withContent(
            new ArrayObject(
                [
                    'application/json'         => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'application/ld+json'      => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'application/vnd.api+json' => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                    'text/html'                => new MediaType(
                        new ArrayObject(['$ref' => '#/components/schemas/Error'])
                    ),
                ]
            )
        );

        return $operation->withResponses(['404' => $response] + $operation->getResponses());
    }
}
