<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\API\Endpoint\Storefront\Address\Validator\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use XCart\API\Endpoint\Storefront\Address\Validator\Constraint\Operation\AddressFieldsTransformer;
use XLite\Core\Database;
use XLite\Model\AddressField;

class RequiredFieldsValidator extends ConstraintValidator
{
    /**
     * @param \XCart\API\Entity\Storefront\Address $value
     */
    public function validate($value, Constraint $constraint)
    {
        $isCreateAction = !$value->getId();

        $requestFields  = AddressFieldsTransformer::transform($value->getFields());
        $requiredFields = $this->getRequiredFields($isCreateAction, $requestFields);

        foreach ($requiredFields as $requiredFieldName) {
            $requestFieldValue = $requestFields[$requiredFieldName] ?? null;

            if ($requestFieldValue === null) {
                $this->context->buildViolation($constraint->missingFieldMessage)
                    ->setParameter('{{ value }}', $this->formatValue($requiredFieldName))
                    ->addViolation();
            } elseif (
                $requestFieldValue === ''
                && !in_array($requiredFieldName, ['state_id', 'custom_state'])
            ) {
                $this->context->buildViolation($constraint->emptyFieldMessage)
                    ->setParameter('{{ value }}', $this->formatValue($requiredFieldName))
                    ->addViolation();
            }
        }
    }

    private function getRequiredFields(bool $isCreateAction, array $requestFields): array
    {
        $requiredFields = [];

        $countryAndStateFields = [
            'country_code',
            'state_id',
            'custom_state',
        ];

        if ($isCreateAction) {
            $requiredFieldsForCreateAction = Database::getRepo(AddressField::class)
                ->createPureQueryBuilder('af')
                ->select('af.serviceName')
                ->andWhere('af.enabled = 1')
                ->andWhere('af.required = 1')
                ->andWhere('af.serviceName NOT IN (:excludedFields)')
                ->setParameter('excludedFields', array_merge($countryAndStateFields, ['email']))
                ->getQuery()->getSingleColumnResult();

            $requiredFields = array_merge(
                $countryAndStateFields,
                $requiredFieldsForCreateAction
            );
        } else {
            foreach ($countryAndStateFields as $field) {
                if (isset($requestFields[$field])) {
                    $requiredFields = $countryAndStateFields;
                    break;
                }
            }
        }

        return $requiredFields;
    }
}
