<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XCart\API\Endpoint\Storefront\Address\Validator\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use XCart\API\Endpoint\Storefront\Address\Validator\Constraint\Operation\AddressFieldsTransformer;
use XLite\Core\Database;
use XLite\Model\Country;
use XLite\Model\State;

class CountryStateCompatibilityValidator extends ConstraintValidator
{
    /**
     * @param \XCart\API\Entity\Storefront\Address $value
     */
    public function validate($value, Constraint $constraint)
    {
        $requestFields = AddressFieldsTransformer::transform($value->getFields());

        if ($this->shouldSkipValidation($requestFields)) {
            return;
        }

        /** @var Country|null $country */
        $country = Database::getRepo(Country::class)->findOneBy([
            'enabled' => 1,
            'code'    => $requestFields['country_code'],
        ]);

        if ($country && $country->isForcedCustomState()) {
            $this->validateCustomStateField($requestFields, $constraint);
        } else {
            $this->validateCountryAndStateFields($requestFields, $country, $constraint);
        }
    }

    private function shouldSkipValidation(array $requestFields): bool
    {
        return !isset(
            $requestFields['country_code'],
            $requestFields['state_id'],
            $requestFields['custom_state']
        );
    }

    private function validateCustomStateField(
        array $requestFields,
        Constraint $constraint
    ): void {
        if ($requestFields['custom_state'] === '') {
            $this->context->buildViolation($constraint->missingCustomStateMessage)
                ->setParameter(
                    '{{ country }}',
                    $this->formatValue($requestFields['country_code'])
                )
                ->addViolation();
        }
    }

    private function validateCountryAndStateFields(
        array $requestFields,
        ?Country $country,
        Constraint $constraint
    ): void {
        $stateId = $requestFields['state_id'];
        $state   = Database::getRepo(State::class)->find($stateId);

        if (!$this->hasCompatibleCountryState($country, $state)) {
            $this->context->buildViolation($constraint->nonCompatibleStateMessage)
                ->setParameter(
                    '{{ country }}',
                    $this->formatValue($requestFields['country_code'])
                )
                ->setParameter(
                    '{{ state }}',
                    $this->formatValue($stateId)
                )
                ->addViolation();
        }
    }

    private function hasCompatibleCountryState(?Country $country, ?State $state): bool
    {
        if (!$country) {
            return false;
        }

        $hasCountryStates = $country->hasStates();

        if (
            $hasCountryStates
            && $state
            && $country->getCode() === $state->getCountry()->getCode()
        ) {
            return true;
        }

        if (
            !$hasCountryStates
            && !$state
        ) {
            return true;
        }

        return false;
    }
}
