<?php

namespace XCart\Extender\Tests\Domain;

use PHPUnit\Framework\TestCase;
use XCart\Extender\Action\FQNValidator;
use XCart\Extender\Domain\SourceMap;
use XCart\Extender\Exception\LogicException;

/**
 * @covers \XCart\Extender\Domain\SourceMap
 */
class SourceMapTest extends TestCase
{
    /**
     * @var SourceMap|null
     */
    protected static ?SourceMap $sourceMap = null;

    public static function setUpBeforeClass(): void
    {
        $sourceMap = new SourceMap(new FQNValidator());

        $sourceMap['XC\\TestModule1'] = '/root/modules/XC/TestModule1/src';
        $sourceMap['XC\\TestModule2'] = '/root/modules/XC/TestModule2/src';
        $sourceMap['XC\\TestModule3'] = '/root/modules/XC/TestModule3/src';
        $sourceMap['XLite\\Modules\\XC\\TestModule1'] = '/root/XLite/Modules/XC/TestModule1';
        $sourceMap['XLite\\Modules\\XC\\TestModule2'] = '/root/XLite/Modules/XC/TestModule2';
        $sourceMap['XLite\\Modules\\XC\\TestModule3'] = '/root/XLite/Modules/XC/TestModule3';
        $sourceMap['XLite'] = '/root/XLite';

        static::$sourceMap = $sourceMap;
    }

    /**
     * @param string $expected
     * @param string $path
     *
     * @dataProvider getTestSuggestFQNByPathProvider
     * @covers \XCart\Extender\Domain\SourceMap::suggestFqnByPath
     */
    public function testSuggestFQNByPath(string $expected, string $path): void
    {
        self::assertEquals($expected, static::$sourceMap->suggestFqnByPath($path));
    }

    public function getTestSuggestFQNByPathProvider(): array
    {
        return [
            ['XC\\TestModule1\\Class1', '/root/modules/XC/TestModule1/src/Class1.php'],
            ['XC\\TestModule1\\Class2', '/root/modules/XC/TestModule1/src/Class2.php'],
            ['XC\\TestModule1\\Package1\\Class1', '/root/modules/XC/TestModule1/src/Package1/Class1.php'],
            ['XLite\\Modules\\XC\\TestModule1\\Class1', '/root/XLite/Modules/XC/TestModule1/Class1.php'],
            ['XLite\\Modules\\XC\\TestModule1\\Class2', '/root/XLite/Modules/XC/TestModule1/Class2.php'],
            ['XLite\\Modules\\XC\\TestModule1\\Package1\\Class1', '/root/XLite/Modules/XC/TestModule1/Package1/Class1.php'],
            ['', '/root/XC/TestModule4/Package1/Class1.php'],
            ['XLite\\Logger', '/root/XLite/Logger.php'],
        ];
    }

    /**
     * @param string $expected
     * @param string $path
     *
     * @throws LogicException
     *
     * @dataProvider getTestSuggestPathByFQNProvider
     * @covers       \XCart\Extender\Domain\SourceMap::suggestPathByFqn
     */
    public function testSuggestPathByFQN(string $expected, string $fqn): void
    {
        self::assertEquals($expected, static::$sourceMap->suggestPathByFqn($fqn));
    }

    public function getTestSuggestPathByFQNProvider(): array
    {
        return [
            ['/root/XLite.php', 'XLite'],
            ['/root/XLite/Logger.php', 'XLite\\Logger'],
            ['/root/XLite/Modules/XC/TestModule1/Class1.php', 'XLite\\Modules\\XC\\TestModule1\\Class1'],
            ['/root/XLite/Modules/XC/TestModule1/Class2.php', 'XLite\\Modules\\XC\\TestModule1\\Class2'],
            ['/root/XLite/Modules/XC/TestModule1/Package1/Class1.php', 'XLite\\Modules\\XC\\TestModule1\\Package1\\Class1'],
            ['/root/modules/XC/TestModule1/src/Class1.php', 'XC\\TestModule1\\Class1'],
            ['/root/modules/XC/TestModule1/src/Class2.php', 'XC\\TestModule1\\Class2'],
            ['/root/modules/XC/TestModule1/src/Package1/Class1.php', 'XC\\TestModule1\\Package1\\Class1'],
        ];
    }
}
