<?php

namespace XCart\Extender\Tests\Action;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use XCart\Extender\Action\Reflector;
use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Mapping\Extender\Mixin;
use XCart\Extender\Mapping\Extender\After;
use XCart\Extender\Mapping\Extender\Before;
use XCart\Extender\Mapping\Extender\Depend;
use XCart\Extender\Domain\EnabledModulesMap;
use XCart\Extender\Mapping\Extender\Rely;
use XCart\Extender\Model\Entity;
use XCart\Extender\Model\Reflection;
use XCart\Extender\Factory\ReflectionFactoryInterface;

/**
 * @covers \XCart\Extender\Action\Reflector
 */
class ReflectorTest extends TestCase
{
    /**
     * @param bool       $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestIsClassProvider
     */
    public function testIsClass(bool $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->isClass(new Entity('')));
    }

    public function getTestIsClassProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setKind('class');

        $case1 = [
            true,
            $reflection1,
        ];

        return [
            $case1,
        ];
    }

    /**
     * @param string     $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestGetParentProvider
     */
    public function testGetParent(string $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->getParent(new Entity('')));
    }

    public function getTestGetParentProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setNamespace('XCart\\Extender\\Tests\\Action');
        $reflection1->setImports([
            'parentclass' => 'Some\\ParentClass',
        ]);
        $reflection1->setParent('ParentClass');

        $case1 = [
            'Some\\ParentClass',
            $reflection1,
        ];

        $reflection2 = new Reflection();
        $reflection2->setNamespace('XCart\\Extender\\Tests\\Action');
        $reflection2->setImports([]);
        $reflection2->setParent('ParentClass');

        $case2 = [
            'XCart\\Extender\\Tests\\Action\\ParentClass',
            $reflection2,
        ];

        $reflection3 = new Reflection();
        $reflection3->setNamespace('XCart\\Extender\\Tests\\Action');
        $reflection3->setImports([]);
        $reflection3->setParent('\\Some\\ParentClass');

        $case3 = [
            'Some\\ParentClass',
            $reflection3,
        ];

        return [
            $case1,
            $case2,
            $case3,
        ];
    }

    /**
     * @param bool       $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestIsMixinProvider
     */
    public function testIsMixin(bool $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->isMixin(new Entity('')));
    }

    public function getTestIsMixinProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setNamespace('XCart\\Extender\\Tests\\Action');

        $reflection1->setAnnotations([new Mixin()]);

        $case1 = [
            true,
            $reflection1,
        ];

        return [
            $case1,
        ];

    }

    /**
     * @param bool       $expected
     * @param Reflection $reflection
     * @param string     $class
     *
     * @throws EntityException
     *
     * @dataProvider getHasAnnotationOfClassProvider
     */
    public function testHasAnnotationOfClass(bool $expected, Reflection $reflection, string $class): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->hasAnnotationOfClass(new Entity(''), $class));

    }

    public function getHasAnnotationOfClassProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Depend(['value' => ['Some\\Module1']]),
        ]);

        $case1 = [
            true,
            $reflection1,
            Depend::class,
        ];

        $case2 = [
            false,
            $reflection1,
            After::class,
        ];

        return [
            $case1,
            $case2,
        ];
    }

    /**
     * @param array      $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestGetDependenciesProvider
     */
    public function testGetDependencies(array $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->getDependencies(new Entity('')));
    }

    public function getTestGetDependenciesProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Depend(['value' => ['Some\\Module1', 'Some\\Module2', '!Some\\Module3']]),
            new Rely(['value' => ['Some\\Module4', 'Some\\Module5', '!Some\\Module6']]),
        ]);

        $case1 = [
            ['Some\\Module1', 'Some\\Module2', 'Some\\Module4', 'Some\\Module5'],
            $reflection1,
        ];

        return [
            $case1,
        ];
    }

    /**
     * @param array      $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestGetIncompatiblesProvider
     */
    public function testGetIncompatibles(array $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->getIncompatibles(new Entity('')));
    }

    public function getTestGetIncompatiblesProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Depend(['value' => ['Some\\Module1', 'Some\\Module2', '!Some\\Module3']]),
            new Rely(['value' => ['Some\\Module4', 'Some\\Module5', '!Some\\Module6']]),
        ]);

        $case1 = [
            ['Some\\Module3', 'Some\\Module6'],
            $reflection1,
        ];

        return [
            $case1,
        ];
    }

    /**
     * @param array      $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestGetBeforeModulesProvider
     */
    public function testGetBeforeModules(array $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->getBeforeModules(new Entity('')));
    }

    public function getTestGetBeforeModulesProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Before(['value' => ['Some\\Module1', 'Some\\Module2']]),
        ]);

        $case1 = [
            ['Some\\Module1', 'Some\\Module2'],
            $reflection1,
        ];

        return [
            $case1,
        ];
    }

    /**
     * @param array      $expected
     * @param Reflection $reflection
     *
     * @throws EntityException
     *
     * @dataProvider getTestGetAfterModulesProvider
     */
    public function testGetAfterModules(array $expected, Reflection $reflection): void
    {
        $reflector = $this->getReflector($reflection, []);

        self::assertEquals($expected, $reflector->getAfterModules(new Entity('')));
    }

    public function getTestGetAfterModulesProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Depend(['value' => ['Some\\Module1', 'Some\\Module2', '!Some\\Module3']]),
            new After(['value' => ['Some\\Module4', 'Some\\Module5']]),
        ]);

        $case1 = [
            ['Some\\Module1', 'Some\\Module2', 'Some\\Module4', 'Some\\Module5'],
            $reflection1,
        ];

        return [
            $case1,
        ];
    }

    /**
     * @param bool       $expected
     * @param Reflection $reflection
     * @param array      $enabledModules
     *
     * @throws EntityException
     *
     * @dataProvider getTestIsCompatibleProvider
     */
    public function testIsCompatible(bool $expected, Reflection $reflection, array $enabledModules): void
    {
        $reflector = $this->getReflector($reflection, $enabledModules);

        self::assertEquals($expected, $reflector->isCompatible(new Entity('')));
    }

    public function getTestIsCompatibleProvider(): array
    {
        $reflection1 = new Reflection();
        $reflection1->setAnnotations([
            new Depend(['value' => ['Some\\Module1', 'Some\\Module2']]),
            new Rely(['value' => ['Some\\Module3', 'Some\\Module4']]),
        ]);

        $case1 = [
            true,
            $reflection1,
            ['Some\\Module1', 'Some\\Module2', 'Some\\Module3', 'Some\\Module4'],
        ];

        $reflection2 = new Reflection();
        $reflection2->setAnnotations([
            new Depend(['value' => ['Some\\Module1', '!Some\\Module2']]),
            new Rely(['value' => ['Some\\Module3', '!Some\\Module4']]),
        ]);

        $case2 = [
            false,
            $reflection2,
            ['Some\\Module1', 'Some\\Module2', 'Some\\Module3', 'Some\\Module4'],
        ];

        return [
            $case1,
            $case2,
        ];
    }

    /**
     * @param string $expected
     * @param string $fqn
     *
     * @dataProvider getTestGetModuleProvider
     */
    public function testGetModule(string $expected, string $fqn): void
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflectionFactory = $this->createMock(ReflectionFactoryInterface::class);

        /** @var MockObject|EnabledModulesMap $reflectionFactory */
        $enabledModulesMap = $this->createMock(EnabledModulesMap::class);

        $reflector = new Reflector($enabledModulesMap, $reflectionFactory);

        self::assertEquals($expected, $reflector->getModule(new Entity($fqn)));
    }

    public function getTestGetModuleProvider(): array
    {
        return [
            [
                'Some\\Module',
                'Some\\Module\\Some\\Class',
            ],
            [
                '',
                'XLite\\View\\Some\\Module',
            ],
        ];
    }

    protected function getReflector(Reflection $reflection, array $enabledModules): ReflectorInterface
    {
        /** @var MockObject|ReflectionFactoryInterface $reflectionFactory */
        $reflectionFactory = $this->createMock(ReflectionFactoryInterface::class);
        $reflectionFactory->method('build')
            ->withAnyParameters()
            ->willReturn($reflection);

        $enabledModulesMap = new EnabledModulesMap();
        $enabledModulesMap->setHashMap($enabledModules);

        return new Reflector($enabledModulesMap, $reflectionFactory);
    }
}
