<?php

namespace XCart\Extender\Tests\Action;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use XCart\Extender\Action\AnnotationsParser;
use XCart\Extender\Action\AnnotationsParserInterface;
use XCart\Extender\Action\Parser;
use XCart\Extender\Exception\ParserException;

/**
 * @covers \XCart\Extender\Action\Parser
 */
class ParserTest extends TestCase
{
    /**
     * @param array  $expected
     * @param string $source
     *
     * @throws ParserException
     *
     * @dataProvider getTestParseSourceProvider
     */
    public function testParseSource(array $expected, string $source): void
    {
        /** @var MockObject|AnnotationsParser $annotationParser */
        $annotationParser = $this->createMock(AnnotationsParserInterface::class);

        $annotationParser->method('parseAnnotations')
            ->withAnyParameters()
            ->willReturn([]);

        $parser = new Parser($annotationParser);

        $reflection = $parser->parseSource($source);

        self::assertEquals($expected['namespace'], $reflection->getNamespace());
        self::assertEquals($expected['imports'], $reflection->getImports());
        self::assertEquals($expected['docComment'], $reflection->getDocComment());
        self::assertEquals($expected['modifier'], $reflection->getModifier());
        self::assertEquals($expected['kind'], $reflection->getKind());
        self::assertEquals($expected['name'], $reflection->getName());
        self::assertEquals($expected['parent'], $reflection->getParent());
        self::assertEquals($expected['interfaces'], $reflection->getInterfaces());
    }

    public function getTestParseSourceProvider(): array
    {
        return [
            [
                [
                    'namespace'  => 'XCart\Extender\Tests\Action\ParserData',
                    'imports'    => [
                        'parentclass'     => 'Some\ParentClass',
                        'parentinterface' => 'Some\ParentInterface',
                        'someclass1'      => 'Some\Class1',
                    ],
                    'docComment' => "/**\n * SomeDocComment\n */",
                    'modifier'   => 'final',
                    'kind'       => 'class',
                    'name'       => 'Class1',
                    'parent'     => 'ParentClass',
                    'interfaces' => ['ParentInterface', 'OtherParentInterface'],
                ],
                file_get_contents(__DIR__ . '/ParserData/Class1.php.test'),
            ],
        ];
    }

    /**
     * @param string $expected
     * @param string $source
     *
     * @throws ParserException
     *
     * @dataProvider getTestParseSourceParserExceptionProvider
     */
    public function testParseSourceParserException(string $expected, string $source): void
    {
        $this->expectException(ParserException::class);
        $this->expectExceptionMessage($expected);

        /** @var MockObject|AnnotationsParser $annotationParser */
        $annotationParser = $this->createMock(AnnotationsParserInterface::class);

        $annotationParser->method('parseAnnotations')
            ->withAnyParameters()
            ->willReturn([]);

        $parser = new Parser($annotationParser);

        $parser->parseSource($source);
    }

    public function getTestParseSourceParserExceptionProvider(): array
    {
        return [
            [
                'Parsing error (namespace)',
                file_get_contents(__DIR__ . '/ParserData/WrongClass1Namespace.php.test'),
            ],
            [
                'Parsing error (imports)',
                file_get_contents(__DIR__ . '/ParserData/WrongClass1Imports.php.test'),
            ],
            [
                'Parsing error (name)',
                file_get_contents(__DIR__ . '/ParserData/WrongClass1Name.php.test'),
            ],
            [
                'Parsing error (parent)',
                file_get_contents(__DIR__ . '/ParserData/WrongClass1Parent.php.test'),
            ],
            [
                'Parsing error (interfaces)',
                file_get_contents(__DIR__ . '/ParserData/WrongClass1Interfaces.php.test'),
            ],
        ];
    }
}
