<?php

namespace XCart\Extender\Tests\Action;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use XCart\Extender\Action\MixinSorter;
use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Model\Entity;
use XCart\Extender\Domain\ModuleDependenciesMap;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;

/**
 * @covers \XCart\Extender\Action\MixinSorter
 */
class MetaDecoratorsSorterTest extends TestCase
{
    /**
     * @param array                 $expected
     * @param ReflectorInterface    $reflector
     * @param ModuleDependenciesMap $moduleDependenciesMap
     * @param array                 $metaDecorators
     *
     * @throws LogicException
     * @throws EntityException
     *
     * @dataProvider getTestSortProvider
     */
    public function testSort(
        array $expected,
        ReflectorInterface $reflector,
        ModuleDependenciesMap $moduleDependenciesMap,
        array $metaDecorators
    ): void {
        $sorter = new MixinSorter(
            $moduleDependenciesMap,
            $reflector
        );

        static::assertEquals($expected, $sorter->sort($metaDecorators));
    }

    public function getTestSortProvider(): array
    {
        return [
            $this->buildTestCase([], [], []),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module2\Class1',
                    'XLite\Module\Some\Module1\Class2',
                    'XLite\Module\Some\Module1\Class1',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => [],
                    'Some\Module2' => [],
                ]
            ),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module1\Class1',
                    'XLite\Module\Some\Module2\Class1',
                    'XLite\Module\Some\Module1\Class2',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => ['Some\Module2'],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => [],
                    'Some\Module2' => [],
                ]
            ),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module1\Class2',
                    'XLite\Module\Some\Module1\Class1',
                    'XLite\Module\Some\Module2\Class1',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => [],
                    'Some\Module2' => ['Some\Module1'],
                ]
            ),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module2\Class1',
                    'XLite\Module\Some\Module1\Class1',
                    'XLite\Module\Some\Module1\Class2',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => ['Some\Module1'],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => [],
                    'Some\Module2' => [],
                ]
            ),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module1\Class2',
                    'XLite\Module\Some\Module1\Class1',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => ['Some\Module2'],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                ]
            ),
        ];
    }

    /**
     * @param array                 $expected
     * @param ReflectorInterface    $reflector
     * @param ModuleDependenciesMap $moduleDependenciesMap
     * @param array                 $metaDecorators
     *
     * @throws EntityException
     * @throws LogicException
     *
     * @dataProvider getTestSortCircularDependencyExceptionProvider
     */
    public function testSortCircularDependencyException(
        array $expected,
        ReflectorInterface $reflector,
        ModuleDependenciesMap $moduleDependenciesMap,
        array $metaDecorators
    ): void {

        $this->expectException(LogicException::class);

        $sorter = new MixinSorter(
            $moduleDependenciesMap,
            $reflector
        );

        $sorter->sort($metaDecorators);
    }

    public function getTestSortCircularDependencyExceptionProvider(): array
    {
        return [
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module1\Class2',
                    'XLite\Module\Some\Module1\Class1',
                    'XLite\Module\Some\Module2\Class1',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => [],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => ['Some\Module1'],
                    'Some\Module2' => ['Some\Module1'],
                ]
            ),
            $this->buildTestCase(
                [
                    'XLite\Module\Some\Module1\Class2',
                    'XLite\Module\Some\Module1\Class1',
                    'XLite\Module\Some\Module2\Class1',
                ],
                [
                    'XLite\Module\Some\Module1\Class1' => [
                        'module' => 'Some\Module1',
                        'before' => ['Some\Module2'],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module1\Class2' => [
                        'module' => 'Some\Module1',
                        'before' => [],
                        'after'  => [],
                    ],
                    'XLite\Module\Some\Module2\Class1' => [
                        'module' => 'Some\Module2',
                        'before' => ['Some\Module1'],
                        'after'  => [],
                    ],
                ],
                [
                    'Some\Module1' => [],
                    'Some\Module2' => [],
                ]
            )
        ];
    }

    public function buildTestCase(array $expected, array $data, array $modules): array
    {
        $entities                  = [];
        $reflectorGetModule        = [];
        $reflectorGetBeforeModules = [];
        $reflectorGetAfterModules  = [];

        foreach ($data as $fqn => $entityData) {
            $entity         = new Entity($fqn);
            $entities[$fqn] = $entity;

            $reflectorGetModule[]        = [$entity, $entityData['module']];
            $reflectorGetBeforeModules[] = [$entity, $entityData['before']];
            $reflectorGetAfterModules[]  = [$entity, $entityData['after']];
        }

        /** @var MockObject|ReflectorInterface $reflector */
        $reflector = $this->createMock(ReflectorInterface::class);
        $reflector->method('getModule')->willReturnMap($reflectorGetModule);
        $reflector->method('getBeforeModules')->willReturnMap($reflectorGetBeforeModules);
        $reflector->method('getAfterModules')->willReturnMap($reflectorGetAfterModules);

        /** @var MockObject|ModuleDependenciesMap $moduleDependencies */
        $moduleDependencies = $this->createMock(ModuleDependenciesMap::class);
        $moduleDependencies->method('offsetGet')->willReturnCallback(function ($offset) use ($modules) {
            return $modules[$offset];
        });

        $moduleDependencies->method('offsetExists')->willReturnCallback(function ($offset) use ($modules) {
            return array_key_exists($offset, $modules);
        });

        $metaDecorators = array_values($entities);

        $expected = array_map(static function ($fqn) use ($entities) {
            return $entities[$fqn];
        }, $expected);

        return [$expected, $reflector, $moduleDependencies, $metaDecorators];
    }
}
