<?php

namespace XCart\Extender\Tests\Action;

use Doctrine\Common\Annotations\AnnotationException;
use Doctrine\Common\Annotations\DocParser;
use PHPUnit\Framework\TestCase;
use ReflectionException;
use XCart\Extender\Action\AnnotationsParserInterface;
use XCart\Extender\Action\ImportedAnnotationsParser;
use XCart\Extender\Mapping\Extender\After;
use XCart\Extender\Mapping\Extender\Depend;
use XCart\Extender\Model\Reflection;

/**
 * @covers \XCart\Extender\Action\ImportedAnnotationsParser
 */
class ImportedAnnotationsParserTest extends TestCase
{
    /**
     * @var AnnotationsParserInterface
     */
    protected static AnnotationsParserInterface $annotationParser;

    public static function setUpBeforeClass(): void
    {
        $docParser = new DocParser();

        static::$annotationParser = new ImportedAnnotationsParser($docParser);
    }

    /**
     * @param array  $expected
     * @param string $docComment
     * @param array  $imports
     *
     * @throws AnnotationException
     * @throws ReflectionException
     *
     * @dataProvider getTestParseAnnotationsProvider
     */
    public function testParseAnnotations(array $expected, string $docComment, array $imports): void
    {
        $reflection = new Reflection();
        $reflection->setDocComment($docComment);

        $reflection->setImports($imports);

        $annotations = static::$annotationParser->parseAnnotations($reflection);

        [$count, $instances] = $expected;

        self::assertCount($count, $annotations);

        foreach ($annotations as $annotation) {
            self::assertContains(get_class($annotation), $instances);
        }
    }

    public function getTestParseAnnotationsProvider(): array
    {
        return [
            [
                [
                    1,
                    [
                        Depend::class,
                    ],
                ],
                "/**\n * @Depend(\"XC\\Module\")\n */",
                [
                    'depend' => Depend::class,
                ],
            ],
            [
                [
                    1,
                    [
                        Depend::class,
                    ],
                ],
                "/**\n * @Depend(\"XC\\Module\")\n * @Missing(\"XC\\Module\")\n */",
                [
                    'depend' => Depend::class,
                ],
            ],
            [
                [
                    2,
                    [
                        Depend::class,
                        After::class,
                    ],
                ],
                "/**\n * @Depend(\"XC\\Module\")\n * @After(\"XC\\Module\")\n */",
                [
                    'depend' => Depend::class,
                    'after'  => After::class,
                ],
            ],
        ];
    }
}
