<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender;

use Closure;
use Doctrine\Common\Annotations\DocParser;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Filesystem\Filesystem;
use XCart\Extender\Action\AnnotationsParserInterface;
use XCart\Extender\Action\ClassCacheBuilder\Builder;
use XCart\Extender\Action\ClassCacheBuilder\BuilderCached;
use XCart\Extender\Action\ClassCacheBuilder\BuilderInterface;
use XCart\Extender\Action\ClassCacheBuilder\ChainBuilder;
use XCart\Extender\Action\ClassCacheBuilder\ChainBuilderCached;
use XCart\Extender\Action\ClassCacheBuilder\ChainBuilderInterface;
use XCart\Extender\Action\ClassCacheBuilder\EntityBuilder;
use XCart\Extender\Action\Copier;
use XCart\Extender\Action\FQNValidator;
use XCart\Extender\Action\ImportedAnnotationsParser;
use XCart\Extender\Action\MixinLookup;
use XCart\Extender\Action\MixinLookupCached;
use XCart\Extender\Action\MixinLookupInterface;
use XCart\Extender\Action\MixinSorter;
use XCart\Extender\Action\Parser;
use XCart\Extender\Action\ParserInterface;
use XCart\Extender\Action\Reader;
use XCart\Extender\Action\ReaderCached;
use XCart\Extender\Action\ReaderInterface;
use XCart\Extender\Action\Reflector;
use XCart\Extender\Action\ReflectorCached;
use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Action\ViewListReader\EntityViewListReader;
use XCart\Extender\Action\ViewListReader\ViewListExtractor;
use XCart\Extender\Action\ViewListReader\ViewListExtractorCached;
use XCart\Extender\Action\ViewListReader\ViewListReader;
use XCart\Extender\Action\Writer;
use XCart\Extender\Autoloader\Development\AncestorStreamFilter;
use XCart\Extender\Autoloader\Development\CommonStreamFilter;
use XCart\Extender\Autoloader\Development\StreamWrapper;
use XCart\Extender\Domain\EnabledModulesMap;
use XCart\Extender\Domain\ModuleDependenciesMap;
use XCart\Extender\Domain\SourceMap;
use XCart\Extender\Domain\SourceMapCached;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetMap;
use XCart\Extender\Domain\TargetStreamMap;
use XCart\Extender\Event\ApiPlatformSubscriber;
use XCart\Extender\Event\DependenciesSubscriber;
use XCart\Extender\Event\DevelopmentModeSubscriber;
use XCart\Extender\Event\DoctrineSubscriber;
use XCart\Extender\Event\ViewListSubscriber;
use XCart\Extender\Factory\CodeFileFactory;
use XCart\Extender\Factory\EntityFactory;
use XCart\Extender\Factory\EntityFactoryCached;
use XCart\Extender\Factory\EntityFactoryInterface;
use XCart\Extender\Factory\ReflectionFactory;
use XCart\Extender\Factory\ReflectionFactoryCached;
use XCart\Extender\Factory\ReflectionFactoryInterface;

class Locator
{
    /**
     * @var array
     */
    private array $services = [];

    public function getBuilder(): BuilderInterface
    {
        return $this->getService(BuilderInterface::class, function () {
            $sourceMap = $this->getSourceMap();
            $builder   = new Builder($sourceMap, $this->getEntityFactory(), $this->getEntityBuilder());

            return new BuilderCached($sourceMap, $this->getTargetStreamMap(), $builder);
        });
    }

    public function getSourceMap(): SourceMapInterface
    {
        return $this->getService(SourceMapInterface::class, static function () {
            $sourceMap = new SourceMap(new FQNValidator());

            return new SourceMapCached($sourceMap);
        });
    }

    public function getEntityFactory(): EntityFactoryInterface
    {
        return $this->getService(EntityFactoryInterface::class, function () {
            $entityFactory = new EntityFactory($this->getSourceMap(), $this->getTargetMap());

            return new EntityFactoryCached($entityFactory);
        });
    }

    public function getTargetMap(): TargetMap
    {
        return $this->getService(TargetMap::class, static function () {
            return new TargetMap();
        });
    }

    public function getEntityBuilder(): EntityBuilder
    {
        return $this->getService(EntityBuilder::class, function () {
            return new EntityBuilder($this->getEntityFactory(), $this->getCodeFileFactory(), $this->getReflector(), $this->getChainBuilder());
        });
    }

    public function getCodeFileFactory(): CodeFileFactory
    {
        return $this->getService(CodeFileFactory::class, function () {
            return new CodeFileFactory($this->getReader(), $this->getCopier(), $this->getWriter(), $this->getReflectionFactory(), $this->getEventDispatcher());
        });
    }

    public function getReader(): ReaderInterface
    {
        return $this->getService(ReaderInterface::class, static function () {
            $reader = new Reader();

            return new ReaderCached($reader);
        });
    }

    public function getCopier(): Copier
    {
        return $this->getService(Copier::class, function () {
            return new Copier($this->getFileSystem());
        });
    }

    public function getFileSystem(): Filesystem
    {
        return $this->getService(Filesystem::class, static function () {
            return new Filesystem();
        });
    }

    public function getWriter(): Writer
    {
        return $this->getService(Writer::class, function () {
            return new Writer($this->getFileSystem());
        });
    }

    public function getReflectionFactory(): ReflectionFactoryInterface
    {
        return $this->getService(ReflectionFactoryInterface::class, function () {
            $reflectionFactory = new ReflectionFactory($this->getReader(), $this->getParser());

            return new ReflectionFactoryCached($reflectionFactory);
        });
    }

    public function getParser(): ParserInterface
    {
        return $this->getService(ParserInterface::class, function () {
            return new Parser($this->getAnnotationsParser());
        });
    }

    public function getAnnotationsParser(): AnnotationsParserInterface
    {
        return $this->getService(AnnotationsParserInterface::class, static function () {
            return new ImportedAnnotationsParser(new DocParser());
        });
    }

    public function getEventDispatcher(): EventDispatcherInterface
    {
        return $this->getService(EventDispatcherInterface::class, static function () {
            return new EventDispatcher();
        });
    }

    public function getApiPlatformSubscriber(): ApiPlatformSubscriber
    {
        return $this->getService(ApiPlatformSubscriber::class, function () {
            return new ApiPlatformSubscriber($this->getCodeFileFactory());
        });
    }

    public function getDependenciesSubscriber(): DependenciesSubscriber
    {
        return $this->getService(DependenciesSubscriber::class, function () {
            return new DependenciesSubscriber($this->getReflector());
        });
    }

    public function getDoctrineSubscriber(): DoctrineSubscriber
    {
        return $this->getService(DoctrineSubscriber::class, function () {
            return new DoctrineSubscriber($this->getCodeFileFactory(), $this->getReflector());
        });
    }

    public function getViewListSubscriber(): ViewListSubscriber
    {
        return $this->getService(ViewListSubscriber::class, function () {
            return new ViewListSubscriber($this->getCodeFileFactory());
        });
    }

    public function getDevelopmentModeSubscriber(): DevelopmentModeSubscriber
    {
        return $this->getService(DevelopmentModeSubscriber::class, function () {
            $commonStreamFilter = new CommonStreamFilter();
            $commonStreamFilter::setSourceMap($this->getSourceMap());
            $commonStreamFilter::setTargetMap($this->getTargetMap());
            $commonStreamWrapper = new StreamWrapper('xcart.common', $commonStreamFilter);

            $ancestorStreamFilter = new AncestorStreamFilter();
            $ancestorStreamFilter::setSourceMap($this->getSourceMap());
            $ancestorStreamFilter::setTargetMap($this->getTargetMap());
            $ancestorStreamWrapper = new StreamWrapper('xcart.ancestor', $ancestorStreamFilter);

            return new DevelopmentModeSubscriber($this->getEntityFactory(), $this->getMixinLookup(), $commonStreamWrapper, $ancestorStreamWrapper);
        });
    }

    public function getReflector(): ReflectorInterface
    {
        return $this->getService(ReflectorInterface::class, function () {
            $reflector = new Reflector($this->getEnabledModulesMap(), $this->getReflectionFactory());

            return new ReflectorCached($reflector);
        });
    }

    public function getEnabledModulesMap(): EnabledModulesMap
    {
        return $this->getService(EnabledModulesMap::class, static function () {
            return new EnabledModulesMap();
        });
    }

    public function getChainBuilder(): ChainBuilderInterface
    {
        return $this->getService(ChainBuilderInterface::class, function () {
            $chainBuilder = new ChainBuilder($this->getEntityFactory(), $this->getCodeFileFactory(), $this->getMixinLookup(), $this->getEventDispatcher());

            return new ChainBuilderCached($chainBuilder, $this->getTargetStreamMap());
        });
    }

    public function getMixinLookup(): MixinLookupInterface
    {
        return $this->getService(MixinLookupInterface::class, function () {
            $mixinLookup = new MixinLookup($this->getSourceMap(), $this->getEntityFactory(), $this->getReflector(), $this->getMixinSorter(), $this->getEventDispatcher());

            return new MixinLookupCached($mixinLookup, $this->getSourceMap(), $this->getEntityFactory(), $this->getFileSystem());
        });
    }

    public function getMixinSorter(): MixinSorter
    {
        return $this->getService(MixinSorter::class, function () {
            return new MixinSorter($this->getModuleDependenciesMap(), $this->getReflector());
        });
    }

    public function getModuleDependenciesMap(): ModuleDependenciesMap
    {
        return $this->getService(ModuleDependenciesMap::class, static function () {
            return new ModuleDependenciesMap();
        });
    }

    public function getTargetStreamMap(): TargetStreamMap
    {
        return $this->getService(TargetStreamMap::class, static function () {
            return new TargetStreamMap();
        });
    }

    public function getViewListReader(): ViewListReader
    {
        return $this->getService(ViewListReader::class, function () {
            $viewListExtractor    = new ViewListExtractorCached(new ViewListExtractor($this->getReflector(), $this->getEventDispatcher()));
            $entityViewListReader = new EntityViewListReader($this->getReflector(), $viewListExtractor);

            return new ViewListReader($this->getSourceMap(), $this->getEntityFactory(), $entityViewListReader);
        });
    }

    public function getService(string $name, Closure $factory)
    {
        if (!isset($this->services[$name])) {
            $this->services[$name] = $factory();
        }

        return $this->services[$name];
    }
}
