<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Factory;

use stdClass;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use XCart\Extender\Action\Copier;
use XCart\Extender\Action\ReaderInterface;
use XCart\Extender\Action\Writer;
use XCart\Extender\CodeGenerator\DocCommentMutation;
use XCart\Extender\CodeGenerator\MutationInterface;
use XCart\Extender\CodeGenerator\ParentMutation;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\IOException;
use XCart\Extender\Model\Entity;
use XCart\Extender\Model\EntityEvent;

use function array_pop;
use function count;
use function explode;
use function implode;
use function is_array;
use function preg_replace;
use function token_get_all;

class CodeFileFactory
{
    /**
     * @var ReaderInterface
     */
    private ReaderInterface $reader;

    /**
     * @var Copier
     */
    private Copier $copier;

    /**
     * @var Writer
     */
    private Writer $writer;

    /**
     * @var ReflectionFactoryInterface
     */
    private ReflectionFactoryInterface $reflectionFactory;

    /**
     * @var EventDispatcherInterface
     */
    private EventDispatcherInterface $eventDispatcher;

    /**
     * @param ReaderInterface            $reader
     * @param Copier                     $copier
     * @param Writer                     $writer
     * @param ReflectionFactoryInterface $reflectionFactory
     * @param EventDispatcherInterface   $eventDispatcher
     */
    public function __construct(
        ReaderInterface $reader,
        Copier $copier,
        Writer $writer,
        ReflectionFactoryInterface $reflectionFactory,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->reader            = $reader;
        $this->copier            = $copier;
        $this->writer            = $writer;
        $this->reflectionFactory = $reflectionFactory;
        $this->eventDispatcher   = $eventDispatcher;
    }

    /**
     * @param Entity $entity
     *
     * @throws EntityException
     */
    public function buildUnaltered(Entity $entity): void
    {
        $event = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($event, 'build-unaltered');

        $this->copier->copy($entity);
    }

    /**
     * @param Entity $entity
     *
     * @throws EntityException
     */
    public function buildAncestorFromSource(Entity $entity): void
    {
        $event = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($event, 'build-ancestor-from-source');

        $source = $this->getEntitySource($entity);
        if ($mutations = $entity->getMutations()) {
            $source = $this->applyMutations($source, $mutations);
        }

        $this->writer->write($entity, $source);
    }

    /**
     * @param Entity $entity
     *
     * @throws EntityException
     */
    public function buildMixinFromSource(Entity $entity): void
    {
        $event = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($event, 'build-mixin-from-source');

        $source = $this->getEntitySource($entity);
        if ($mutations = $entity->getMutations()) {
            $source = $this->applyMutations($source, $mutations);
        }

        $this->writer->write($entity, $source);
    }

    /**
     * @param Entity $entity
     *
     * @throws EntityException
     */
    public function buildDescendantFromScratch(Entity $entity): void
    {
        $event = new EntityEvent($entity);
        $this->eventDispatcher->dispatch($event, 'build-descendant-from-scratch');

        $source = $this->getEmptySource($entity);
        if ($mutations = $entity->getMutations()) {
            $source = $this->applyMutations($source, $mutations);
        }

        $this->writer->write($entity, $source);
    }

    /**
     * @param Entity $entity
     * @param string $parentFqn
     */
    public function addParentMutation(Entity $entity, string $parentFqn): void
    {
        $entity->addMutation(new ParentMutation($parentFqn));
    }

    /**
     * @param Entity $entity
     * @param array  $annotations
     */
    public function addRemoveAnnotationMutation(Entity $entity, array $annotations): void
    {
        $entity->addMutation(new DocCommentMutation(static function (string $docComment) use ($annotations) {
            foreach ($annotations as $annotation) {
                $docComment = preg_replace("/@({$annotation}\b)/i", ' $1', (string) $docComment);
            }

            return $docComment;
        }));
    }

    /**
     * @param Entity $entity
     * @param array  $annotations
     */
    public function addAddAnnotationMutation(Entity $entity, array $annotations): void
    {
        $entity->addMutation(new DocCommentMutation(static function (string $docComment) use ($annotations) {
            $docCommentParts = explode("\n", $docComment);
            $terminator      = array_pop($docCommentParts);

            foreach ($annotations as $annotation => $params) {
                $docCommentParts[] = " * @{$annotation}";
            }

            $docCommentParts[] = $terminator;

            return implode("\n", $docCommentParts);
        }));
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     */
    private function getEntitySource(Entity $entity): string
    {
        try {
            return $this->reader->readSource($entity->getSourcePath());
        } catch (IOException $exception) {
            throw EntityException::fromReadSource($entity, $exception);
        }
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     */
    private function getEmptySource(Entity $entity): string
    {
        $reflection = $this->reflectionFactory->build($entity);

        $namespace  = $reflection->getNamespace();

        $imports = "\n";
        foreach ($reflection->getImports() as $alias => $import) {
            $imports .= "use {$import} as {$alias};\n";
        }

        $docComment = $reflection->getDocComment();
        $modifier   = $reflection->getModifier();
        $modifier   = $modifier ? ($modifier . ' ') : '';
        $name       = $reflection->getName();
        $parent     = $reflection->getParent() ?: stdClass::class;

        $namespaceStatement = $namespace ? "namespace {$namespace};\n" : '';

        return <<<PHPCODE
<?php
{$namespaceStatement}{$imports}{$docComment}
{$modifier}class {$name} extends {$parent} {}
PHPCODE;
    }

    /**
     * @param string                    $source
     * @param array|MutationInterface[] $mutations
     *
     * @return string
     */
    private function applyMutations(string $source, array $mutations): string
    {
        $tokens = token_get_all($source);

        $position = 0;
        $count    = count($tokens);

        while ($mutations && $position < $count) {
            $token = $tokens[$position];
            [$token, $content] = is_array($token) ? $token : [$token, $token];
            foreach ($mutations as $k => $mutation) {
                if ($mutation->isAnchor($token, $content)) {
                    unset($mutations[$k]);

                    $substitutions = $mutation->getSubstitutions($tokens, $position);

                    foreach ($substitutions as $index => $substitution) {
                        if (is_array($tokens[$index])) {
                            $tokens[$index][1] = $substitution;
                        } else {
                            $tokens[$index] = $substitution;
                        }
                    }
                }
            }

            $position++;
        }

        $result = [];
        foreach ($tokens as $token) {
            $result[] = is_array($token) ? $token[1] : $token;
        }

        return implode('', $result);
    }
}
