<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender;

use Exception;
use XCart\Extender\Action\ClassCacheBuilder\BuilderInterface;
use XCart\Extender\Action\ViewListReader\ViewListReader;
use XCart\Extender\Domain\SourceMap;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetStreamMap;

use function register_shutdown_function;

class Extender
{
    /**
     * @var Locator
     */
    private Locator $locator;

    /**
     * @throws Exception
     */
    public function __construct()
    {
        $this->locator = new Locator();
    }

    /**
     * @param array $sources
     *
     * @return self
     * @throws Exception
     */
    public function setSources(array $sources): self
    {
        /** @var SourceMap $sourceMap */
        $sourceMap = $this->locator->getSourceMap();
        $sourceMap->setHashMap($sources);

        return $this;
    }

    /**
     * @param string $targetRoot
     *
     * @return self
     * @throws Exception
     */
    public function setTargetRoot(string $targetRoot): self
    {
        $targetMap = $this->locator->getTargetMap();
        $targetMap->setTargetRoot($targetRoot);

        return $this;
    }

    /**
     * @param array $modules
     *
     * @return self
     * @throws Exception
     */
    public function setModules(array $modules): self
    {
        $enabledModulesMap = $this->locator->getEnabledModulesMap();
        $enabledModulesMap->setHashMap($modules);

        return $this;
    }

    /**
     * @param array $dependencies
     *
     * @return self
     * @throws Exception
     */
    public function setDependencies(array $dependencies): self
    {
        $moduleDependenciesMap = $this->locator->getModuleDependenciesMap();
        $moduleDependenciesMap->setHashMap($dependencies);

        return $this;
    }

    public function addSubscribers(): self
    {
        $eventDispatcher = $this->locator->getEventDispatcher();

        $eventDispatcher->addSubscriber($this->locator->getApiPlatformSubscriber());
        $eventDispatcher->addSubscriber($this->locator->getDependenciesSubscriber());
        $eventDispatcher->addSubscriber($this->locator->getDoctrineSubscriber());
        $eventDispatcher->addSubscriber($this->locator->getViewListSubscriber());

        return $this;
    }

    /**
     * @return self
     * @throws Exception
     */
    public function setDeveloperMode(): self
    {
        $eventDispatcher = $this->locator->getEventDispatcher();

        $developmentModeSubscriber = $this->locator->getDevelopmentModeSubscriber();

        $eventDispatcher->addSubscriber($developmentModeSubscriber);

        register_shutdown_function(static function () use ($eventDispatcher, $developmentModeSubscriber) {
            $eventDispatcher->removeSubscriber($developmentModeSubscriber);
        });

        return $this;
    }

    /**
     * @return SourceMapInterface
     * @throws Exception
     */
    public function getSourceMap(): SourceMapInterface
    {
        return $this->locator->getSourceMap();
    }

    /**
     * @return BuilderInterface
     * @throws Exception
     */
    public function getBuilder(): BuilderInterface
    {
        return $this->locator->getBuilder();
    }

    /**
     * @return TargetStreamMap
     * @throws Exception
     */
    public function getTargetStreamMap(): TargetStreamMap
    {
        return $this->locator->getTargetStreamMap();
    }

    /**
     * @return ViewListReader
     * @throws Exception
     */
    public function getViewListReader(): ViewListReader
    {
        return $this->locator->getViewListReader();
    }
}
