<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Exception;

use Exception;
use XCart\Extender\Model\Entity;

class EntityException extends Exception
{
    /**
     * @var Entity
     */
    private Entity $entity;

    /**
     * @param Entity         $entity
     * @param Exception|null $exception
     *
     * @return static
     */
    public static function fromReadSource(Entity $entity, ?Exception $exception = null): self
    {
        $exception = new self("Read error ({$entity->getSourcePath()}) for {$entity->getFqn()}", $exception ? $exception->getCode() : 0, $exception);

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @param Entity         $entity
     * @param array|Entity[] $mixins
     *
     * @return static
     */
    public static function fromLockedEntityWithMixins(Entity $entity, array $mixins): self
    {
        $mixinFqn = array_map(static fn (Entity $mixin): string => $mixin->getFqn(), $mixins);

        $exception = new self("Entity {$entity->getFqn()} with source path ({$entity->getSourcePath()}) is Locked but mixins are introduced: " . implode(', ', $mixinFqn));

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @param Entity $entity
     *
     * @return static
     */
    public static function fromFQNToPathEqualityCheck(Entity $entity): self
    {
        $exception = new self("Entity {$entity->getFqn()} not equal to the entity source path ({$entity->getSourcePath()})");

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @param Entity         $entity
     * @param Exception|null $exception
     *
     * @return static
     */
    public static function fromEmptySource(Entity $entity, ?Exception $exception = null): self
    {
        $exception = new self("The source ({$entity->getSourcePath()}) for {$entity->getFqn()} is empty", $exception ? $exception->getCode() : 0, $exception);

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @param Entity         $entity
     * @param Exception|null $exception
     *
     * @return static
     */
    public static function fromParsing(Entity $entity, ?Exception $exception = null): self
    {
        $exception = new self("Source ({$entity->getSourcePath()}) parsing error for {$entity->getFqn()}", $exception ? $exception->getCode() : 0, $exception);

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @param Entity         $entity
     * @param Exception|null $exception
     *
     * @return static
     */
    public static function fromWriteSource(Entity $entity, ?Exception $exception = null): self
    {
        $exception = new self("Write error in source ({$entity->getSourcePath()}) for {$entity->getFqn()}", $exception ? $exception->getCode() : 0, $exception);

        $exception->setEntity($entity);

        return $exception;
    }

    /**
     * @return Entity
     */
    public function getEntity(): Entity
    {
        return $this->entity;
    }

    /**
     * @param Entity $entity
     */
    public function setEntity(Entity $entity): void
    {
        $this->entity = $entity;
    }
}
