<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Event;

use Doctrine\ORM\Mapping\HasLifecycleCallbacks;
use Doctrine\ORM\Mapping\MappedSuperclass;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use XCart\Extender\Action\ReflectorInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Factory\CodeFileFactory;
use XCart\Extender\Model\EntityEvent;

class DoctrineSubscriber implements EventSubscriberInterface
{
    /**
     * @var CodeFileFactory
     */
    private CodeFileFactory $codeFileFactory;

    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @var array
     */
    private array $ancestorHasLifecycleCallbacks = [];

    /**
     * @var array
     */
    private array $addHasLifecycleCallbacksToDescendant = [];

    /**
     * @param CodeFileFactory    $codeFileFactory
     * @param ReflectorInterface $reflector
     */
    public function __construct(
        CodeFileFactory $codeFileFactory,
        ReflectorInterface $reflector
    ) {
        $this->codeFileFactory = $codeFileFactory;
        $this->reflector       = $reflector;
    }

    /**
     * @return string[]
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'build-ancestor-from-source'    => 'onBuildAncestorFromSource',
            'build-mixin-from-source'       => 'onBuildMixinFromSource',
            'build-descendant-from-scratch' => 'onBuildDescendantFromScratch',
        ];
    }

    /**
     * @param EntityEvent $event
     *
     * @throws EntityException
     */
    public function onBuildAncestorFromSource(EntityEvent $event): void
    {
        $entity = $event->getEntity();

        $this->ancestorHasLifecycleCallbacks[$entity->getFqn()]        = $this->reflector->hasAnnotationOfClass($entity, HasLifecycleCallbacks::class);
        $this->addHasLifecycleCallbacksToDescendant[$entity->getFqn()] = false;

        $this->codeFileFactory->addRemoveAnnotationMutation(
            $entity,
            [
                'ORM\\\\Entity',
                'ORM\\\\Table',
                'ORM\\\\Index',
                'ORM\\\\UniqueConstraint',
                'ORM\\\\InheritanceType',
                'ORM\\\\DiscriminatorColumn',
                'ORM\\\\DiscriminatorMap',
                'ORM\\\\MappedSuperclass',
            ]
        );
    }

    /**
     * @param EntityEvent $event
     *
     * @throws EntityException
     */
    public function onBuildMixinFromSource(EntityEvent $event): void
    {
        $entity      = $event->getEntity();
        $ancestorFqn = $entity->getAncestorFqn();

        if (
            $ancestorFqn
            && ($this->ancestorHasLifecycleCallbacks[$ancestorFqn] ?? false) === false
            && $this->reflector->hasAnnotationOfClass($entity, HasLifecycleCallbacks::class)
            && !$this->reflector->hasAnnotationOfClass($entity, MappedSuperclass::class)
        ) {
            $this->addHasLifecycleCallbacksToDescendant[$ancestorFqn] = true;
        }
    }

    /**
     * @param EntityEvent $event
     */
    public function onBuildDescendantFromScratch(EntityEvent $event): void
    {
        $entity      = $event->getEntity();
        $ancestorFqn = $entity->getAncestorFqn();

        if (
            $ancestorFqn
            && ($this->addHasLifecycleCallbacksToDescendant[$ancestorFqn] ?? false) === true
        ) {
            $this->codeFileFactory->addAddAnnotationMutation($entity, ['HasLifecycleCallbacks']);
        }
    }
}
