<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Domain;

use XCart\Extender\Exception\LogicException;

class SourceMapCached implements SourceMapInterface
{
    public array $suggestFqnByPathCache = [];

    public array $suggestPathByFqnCache = [];

    public ?array $filesCache = null;

    public ?array $coreFilesCache = null;

    public ?array $moduleFilesCache = null;

    /**
     * @param SourceMapInterface $sourceMap
     */
    public function __construct(
        private SourceMapInterface $sourceMap
    ) {
    }

    public function suggestFqnByPath(string $filePath): string
    {
        if (!isset($this->suggestFqnByPathCache[$filePath])) {
            $fqn = $this->sourceMap->suggestFqnByPath($filePath);

            $this->suggestFqnByPathCache[$filePath] = $fqn;
            $this->suggestPathByFqnCache[$fqn] = $filePath;
        }

        return $this->suggestFqnByPathCache[$filePath];
    }

    /**
     * @throws LogicException
     */
    public function suggestPathByFqn(string $fqn): string
    {
        if (!isset($this->suggestPathByFqnCache[$fqn])) {
            $filePath = $this->sourceMap->suggestPathByFqn($fqn);

            $this->suggestFqnByPathCache[$filePath] = $fqn;
            $this->suggestPathByFqnCache[$fqn] = $filePath;
        }

        return $this->suggestPathByFqnCache[$fqn];
    }

    public function getFiles(): array
    {
        if ($this->filesCache === null) {
            $this->filesCache = $this->sourceMap->getFiles();
        }

        return $this->filesCache;
    }

    public function getCoreFiles(): array
    {
        if ($this->coreFilesCache === null) {
            $this->coreFilesCache = $this->sourceMap->getCoreFiles();
        }

        return $this->coreFilesCache;
    }

    public function getModuleFiles(): array
    {
        if ($this->moduleFilesCache === null) {
            $this->moduleFilesCache = $this->sourceMap->getModuleFiles();
        }

        return $this->moduleFilesCache;
    }

    public function isFqnOfSource(string $fqn): bool
    {
        return $this->sourceMap->isFqnOfSource($fqn);
    }

    public function setHashMap(array $hashMap): void
    {
        $this->sourceMap->setHashMap($hashMap);
    }
}
