<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Domain;

use AppendIterator;
use CallbackFilterIterator;
use FilesystemIterator;
use Iterator;
use RecursiveCallbackFilterIterator;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use XCart\Extender\Action\FQNValidator;
use XCart\Extender\Exception\LogicException;

use function array_keys;
use function array_merge;
use function iterator_to_array;
use function pathinfo;
use function str_ends_with;
use function str_replace;
use function str_starts_with;
use function strlen;
use function substr;

use const PATHINFO_DIRNAME;

class SourceMap extends HashMapAbstract implements SourceMapInterface
{
    public function __construct(
        private FQNValidator $FQNValidator
    ) {
    }

    public function suggestFqnByPath(string $filePath): string
    {
        foreach ($this as $namespacePrefix => $rootPath) {
            if (str_starts_with($filePath, $rootPath) && str_ends_with($filePath, '.php')) {
                $fqn = $namespacePrefix . str_replace('/', '\\', substr($filePath, strlen($rootPath), -4));

                return $this->FQNValidator->validate($fqn) ? $fqn : '';
            }
        }

        return '';
    }

    /**
     * @throws LogicException
     */
    public function suggestPathByFqn(string $fqn): string
    {
        foreach ($this as $namespacePrefix => $rootPath) {
            if ($fqn === $namespacePrefix) {
                return rtrim($rootPath, '/') . '.php';
            }

            if (str_starts_with($fqn, $namespacePrefix . '\\')) {
                return $rootPath . '/' . str_replace('\\', '/', substr($fqn, strlen($namespacePrefix) + 1)) . '.php';
            }
        }

        throw LogicException::fromEmptySourcePath($fqn);
    }

    public function getFiles(): array
    {
        return array_merge($this->getCoreFiles(), $this->getModuleFiles());
    }

    public function getCoreFiles(): array
    {
        return array_keys(iterator_to_array($this->getCoreFilesIterator()));
    }

    public function getModuleFiles(): array
    {
        return array_keys(iterator_to_array($this->getModuleFilesIterator()));
    }

    public function isFqnOfSource(string $fqn): bool
    {
        if ($fqn === 'XLite' || $fqn === 'XLiteAncestor') {
            return true;
        }

        foreach ($this as $namespacePrefix => $rootPath) {
            if (str_starts_with($fqn, $namespacePrefix . '\\')) {
                return true;
            }
        }

        return false;
    }

    private function getCoreFilesIterator(): Iterator
    {
        $iterator = new AppendIterator();

        foreach ($this as $namespacePrefix => $rootPath) {
            if ($namespacePrefix === 'XLite') {
                $iterator->append($this->getCoreIterator($rootPath));
            }
        }

        return new CallbackFilterIterator(
            $iterator,
            static function ($item) {
                return $item->getExtension() === 'php';
            }
        );
    }

    private function getModuleFilesIterator(): Iterator
    {
        $iterator = new AppendIterator();

        foreach ($this as $namespacePrefix => $rootPath) {
            if ($namespacePrefix !== 'XLite') {
                $iterator->append($this->getModuleIterator($rootPath));
            }
        }

        return new CallbackFilterIterator(
            $iterator,
            static function ($item) {
                return $item->getExtension() === 'php';
            }
        );
    }

    private function getModuleIterator(string $path): Iterator
    {
        $iterator = new RecursiveDirectoryIterator(
            $path,
            FilesystemIterator::SKIP_DOTS | FilesystemIterator::KEY_AS_PATHNAME
        );

        $iterator = new RecursiveCallbackFilterIterator(
            $iterator,
            static function ($item) use ($path) {
                return $item->getType() !== 'dir'
                    || (
                        $item->getRealPath() !== $path . '/lib'
                        && $item->getRealPath() !== $path . '/hooks'
                        && $item->getRealPath() !== $path . '/vendor'
                        && $item->getRealPath() !== $path . '/upgrade'
                        && $item->getRealPath() !== $path . '/Tests'
                    );
            }
        );

        return new RecursiveIteratorIterator($iterator);
    }

    private function getCoreIterator(string $path): Iterator
    {
        $iterator = new RecursiveDirectoryIterator(
            pathinfo($path, PATHINFO_DIRNAME),
            FilesystemIterator::SKIP_DOTS | FilesystemIterator::KEY_AS_PATHNAME
        );

        $iterator = new RecursiveCallbackFilterIterator(
            $iterator,
            static function ($item) use ($path) {
                return $item->getType() !== 'dir'
                    || !str_starts_with($item->getRealPath(), $path . '/Module/');
            }
        );

        return new RecursiveIteratorIterator($iterator);
    }
}
