<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Domain;

use ArrayAccess;
use Iterator;

use function current;
use function key;
use function next;
use function reset;

abstract class HashMapAbstract implements ArrayAccess, Iterator
{
    /**
     * @var array
     */
    protected array $hashMap = [];

    /**
     * @param array $data
     */
    public function merge(array $data): void
    {
        foreach ($data as $key => $value) {
            $this->hashMap[$key] = $value;
        }
    }

    /**
     * @param mixed $offset
     *
     * @return bool
     */
    public function offsetExists(mixed $offset): bool
    {
        return isset($this->hashMap[$offset]);
    }

    /**
     * @param mixed $offset
     *
     * @return mixed
     */
    public function offsetGet(mixed $offset): mixed
    {
        return $this->hashMap[$offset];
    }

    /**
     * @param mixed $offset
     * @param mixed $value
     */
    public function offsetSet(mixed $offset, mixed $value): void
    {
        $this->hashMap[$offset] = $value;
    }

    /**
     * @param mixed $offset
     */
    public function offsetUnset(mixed $offset): void
    {
        unset($this->hashMap[$offset]);
    }

    /**
     * @return false|mixed
     */
    public function current(): mixed
    {
        return current($this->hashMap);
    }

    public function next(): void
    {
        next($this->hashMap);
    }

    /**
     * @return int|string|null
     */
    public function key(): mixed
    {
        return key($this->hashMap);
    }

    /**
     * @return bool
     */
    public function valid(): bool
    {
        return key($this->hashMap) !== null;
    }

    public function rewind(): void
    {
        reset($this->hashMap);
    }

    /**
     * @return array
     */
    public function getHashMap(): array
    {
        return $this->hashMap;
    }

    /**
     * @param array $hashMap
     */
    public function setHashMap(array $hashMap): void
    {
        $this->hashMap = $hashMap;
    }
}
