<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Autoloader;

use XCart\Extender\Action\ClassCacheBuilder\BuilderInterface;
use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetMap;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;

use function ltrim;

class DevelopmentAutoloader extends Autoloader
{
    private BuilderInterface $builder;

    public function __construct(
        BuilderInterface $builder,
        SourceMapInterface $sourceMap,
        TargetMap $targetMap
    ) {
        parent::__construct($sourceMap, $targetMap);

        $this->builder = $builder;
    }

    /**
     * @throws LogicException
     */
    public function classExists(string $class): bool
    {
        $fqn = ltrim($class, '\\');

        if ($this->sourceMap->isFqnOfSource($fqn)) {
            try {
                $file = $this->builder->buildByFQN($fqn);
                if ($file) {
                    return true;
                }
            } catch (EntityException $e) {
                return false;
            }
        }

        return false;
    }

    /**
     * @throws LogicException
     */
    public function autoload(string $class): bool
    {
        $fqn = ltrim($class, '\\');

        if ($this->sourceMap->isFqnOfSource($fqn)) {
            try {
                $file = $this->builder->buildByFQN($fqn);
                if ($file) {
                    include $file;

                    return true;
                }
            } catch (EntityException $e) {
                return false;
            }
        }

        return false;
    }

    public function getTargetStreamPath(string $class): string
    {
        $fqn = ltrim($class, '\\');

        return $this->sourceMap->isFqnOfSource($fqn)
            ? $this->builder->buildByFQN($fqn)
            : '';
    }
}
