<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Autoloader;

use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Domain\TargetMap;

use function ltrim;
use function spl_autoload_register;
use function spl_autoload_unregister;

abstract class Autoloader implements AutoloaderInterface
{
    protected SourceMapInterface $sourceMap;

    protected TargetMap $targetMap;

    public function __construct(
        SourceMapInterface $sourceMap,
        TargetMap $targetMap
    ) {
        $this->sourceMap = $sourceMap;
        $this->targetMap = $targetMap;
    }

    public function register(): void
    {
        spl_autoload_register([$this, 'autoload']);
    }

    public function unregister(): void
    {
        spl_autoload_unregister([$this, 'autoload']);
    }

    public function getSourceFilePath(string $class): string
    {
        $fqn = ltrim($class, '\\');

        if ($this->sourceMap->isFqnOfSource($fqn)) {
            return $this->sourceMap->suggestPathByFqn($fqn);
        }

        return '';
    }

    public function getTargetFilePath(string $class): string
    {
        $file = $this->targetMap->suggestPathByFqn(ltrim($class, '\\'));

        return file_exists($file) ? $file : '';
    }

    public function suggestFqnByPath(string $path): string
    {
        return $this->sourceMap->suggestFqnByPath($path);
    }

    public function getTargetStreamPath(string $class): string
    {
        return $this->getTargetFilePath($class);
    }
}
