<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action\ViewListReader;

use XCart\Extender\Domain\SourceMapInterface;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Exception\LogicException;
use XCart\Extender\Factory\EntityFactoryInterface;

class ViewListReader
{
    /**
     * @var SourceMapInterface
     */
    private SourceMapInterface $sourceMap;

    /**
     * @var EntityFactoryInterface
     */
    private EntityFactoryInterface $entityFactory;

    /**
     * @var EntityViewListReader
     */
    private EntityViewListReader $entityViewListReader;

    /**
     * @param SourceMapInterface     $sourceMap
     * @param EntityFactoryInterface $entityFactory
     * @param EntityViewListReader   $entityViewListReader
     */
    public function __construct(
        SourceMapInterface $sourceMap,
        EntityFactoryInterface $entityFactory,
        EntityViewListReader $entityViewListReader
    ) {
        $this->sourceMap            = $sourceMap;
        $this->entityFactory        = $entityFactory;
        $this->entityViewListReader = $entityViewListReader;
    }

    /**
     * @param string $fqn
     * @param string $sourcePath
     *
     * @return array
     * @throws EntityException
     * @throws LogicException
     */
    public function readByFQN(string $fqn, string $sourcePath = ''): array
    {
        $entity = $this->entityFactory->build($fqn, $sourcePath);

        return $this->entityViewListReader->readByEntity($entity);
    }

    /**
     * @param string $sourcePath
     *
     * @return array
     * @throws EntityException
     * @throws LogicException
     */
    public function readByPath(string $sourcePath): array
    {
        $fqn = $this->sourceMap->suggestFqnByPath($sourcePath);

        return $fqn ? $this->readByFQN($fqn, $sourcePath) : [];
    }
}
