<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use XCart\Extender\Exception\EntityException;
use XCart\Extender\Model\Entity;

class ReflectorCached implements ReflectorInterface
{
    /**
     * @var ReflectorInterface
     */
    private ReflectorInterface $reflector;

    /**
     * @var array
     */
    private array $cache = [];

    /**
     * @param ReflectorInterface $reflector
     */
    public function __construct(
        ReflectorInterface $reflector
    ) {
        $this->reflector = $reflector;
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isFQNEqualToPath(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isFQNEqualToPath($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isClass(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isClass($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isLocked(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isLocked($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     */
    public function getParent(Entity $entity): string
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getParent($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isMixin(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isMixin($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     * @param string $class
     *
     * @return bool
     * @throws EntityException
     */
    public function hasAnnotationOfClass(Entity $entity, string $class): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__][$class])) {
            $this->cache[$fqn][__FUNCTION__][$class] = $this->reflector->hasAnnotationOfClass($entity, $class);
        }

        return $this->cache[$fqn][__FUNCTION__][$class];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getDependencies(Entity $entity): array
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getDependencies($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getIncompatibles(Entity $entity): array
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getIncompatibles($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getBeforeModules(Entity $entity): array
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getBeforeModules($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getAfterModules(Entity $entity): array
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getAfterModules($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getViewListAnnotations(Entity $entity): array
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getViewListAnnotations($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isCompatible(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isCompatible($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     */
    public function getModule(Entity $entity): string
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->getModule($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }

    public function isController(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (!isset($this->cache[$fqn][__FUNCTION__])) {
            $this->cache[$fqn][__FUNCTION__] = $this->reflector->isController($entity);
        }

        return $this->cache[$fqn][__FUNCTION__];
    }
}
