<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use XCart\Extender\Mapping\Extender\Mixin;
use XCart\Extender\Mapping\Extender\After;
use XCart\Extender\Mapping\Extender\Before;
use XCart\Extender\Mapping\Extender\Depend;
use XCart\Extender\Mapping\Extender\Locked;
use XCart\Extender\Mapping\Extender\Rely;
use XCart\Extender\Mapping\ListChild;
use XCart\Extender\Domain\EnabledModulesMap;
use XCart\Extender\Exception\EntityException;
use XCart\Extender\Factory\ReflectionFactoryInterface;
use XCart\Extender\Model\Entity;
use XCart\Extender\Model\Reflection;

use function array_merge;
use function explode;
use function ltrim;
use function strpos;
use function strtolower;
use function substr;

class Reflector implements ReflectorInterface
{
    /**
     * @var EnabledModulesMap
     */
    private EnabledModulesMap $enabledModulesMap;

    /**
     * @var ReflectionFactoryInterface
     */
    private ReflectionFactoryInterface $reflectionFactory;

    /**
     * @param EnabledModulesMap          $enabledModulesMap
     * @param ReflectionFactoryInterface $reflectionFactory
     */
    public function __construct(
        EnabledModulesMap $enabledModulesMap,
        ReflectionFactoryInterface $reflectionFactory
    ) {
        $this->reflectionFactory = $reflectionFactory;
        $this->enabledModulesMap = $enabledModulesMap;
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isFQNEqualToPath(Entity $entity): bool
    {
        $reflection = $this->getReflection($entity);

        $fqnFromSource = $this->buildFqn($reflection->getName(), $reflection->getImports(), $reflection->getNamespace());

        return $fqnFromSource === $entity->getFqn();
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isClass(Entity $entity): bool
    {
        $reflection = $this->getReflection($entity);

        return $reflection->getKind() === 'class';
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isLocked(Entity $entity): bool
    {
        $reflection = $this->getReflection($entity);

        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof Locked) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param Entity $entity
     *
     * @return string
     * @throws EntityException
     */
    public function getParent(Entity $entity): string
    {
        $reflection = $this->getReflection($entity);

        return $this->buildFqn($reflection->getParent(), $reflection->getImports(), $reflection->getNamespace());
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isMixin(Entity $entity): bool
    {
        $reflection = $this->getReflection($entity);

        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof Mixin) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param Entity $entity
     * @param string $class
     *
     * @return bool
     * @throws EntityException
     */
    public function hasAnnotationOfClass(Entity $entity, string $class): bool
    {
        $reflection = $this->getReflection($entity);

        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof $class) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getDependencies(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $dependencies = [];
        foreach ($reflection->getAnnotations() as $annotation) {
            if (
                $annotation instanceof Depend
                || $annotation instanceof Rely
            ) {
                $dependencies[] = $annotation->dependencies;
            }
        }

        return $dependencies ? array_merge(...$dependencies) : [];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getIncompatibles(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $incompatibilities = [];
        foreach ($reflection->getAnnotations() as $annotation) {
            if (
                $annotation instanceof Depend
                || $annotation instanceof Rely
            ) {
                $incompatibilities[] = $annotation->incompatibilities;
            }
        }

        return $incompatibilities ? array_merge(...$incompatibilities) : [];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getBeforeModules(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $modules = [];
        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof Before) {
                $modules[] = $annotation->modules;
            }
        }

        return $modules ? array_merge(...$modules) : [];
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    public function getAfterModules(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $modules = [];
        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof Depend) {
                $modules[] = $annotation->dependencies;
            }
            if ($annotation instanceof After) {
                $modules[] = $annotation->modules;
            }
        }

        return $modules ? array_merge(...$modules) : [];
    }

    /**
     * @param Entity $entity
     *
     * @return ListChild[]
     * @throws EntityException
     */
    public function getViewListAnnotations(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $annotations = [];
        foreach ($reflection->getAnnotations() as $annotation) {
            if ($annotation instanceof ListChild) {
                $annotations[] = $annotation;
            }
        }

        return $annotations;
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     * @throws EntityException
     */
    public function isCompatible(Entity $entity): bool
    {
        $module = $this->getModule($entity);
        $parentModule = $this->isMixin($entity) ? $this->getParentModule($entity) : '';

        return (empty($module) || $this->enabledModulesMap->hasAll([$module]))
            && (empty($parentModule) || $this->enabledModulesMap->hasAll([$parentModule]))
            && $this->enabledModulesMap->isResolved($this->getDependencies($entity), $this->getIncompatibles($entity));
    }

    /**
     * @param Entity $entity
     *
     * @return string
     */
    public function getModule(Entity $entity): string
    {
        $fqnParts = explode('\\', $entity->getFqn());

        $author = $fqnParts[0] ?? '';
        $name   = $fqnParts[1] ?? '';

        return $author !== 'XLite' && $name
            ? "{$author}\\{$name}"
            : '';
    }

    /**
     * @param Entity $entity
     *
     * @return string
     */
    public function getParentModule(Entity $entity): string
    {
        $fqnParts = explode('\\', $this->getParent($entity));

        $author = $fqnParts[0] ?? '';
        $name   = $fqnParts[1] ?? '';

        return $author !== 'XLite' && $name
            ? "{$author}\\{$name}"
            : '';
    }

    /**
     * @param Entity $entity
     *
     * @return bool
     */
    public function isController(Entity $entity): bool
    {
        $fqn = $entity->getFqn();

        if (
            strpos($fqn, 'XLite\\Controller\\Admin') === 0
            || strpos($fqn, 'XLite\\Controller\\Customer') === 0
        ) {
            return true;
        }

        $module = $this->getModule($entity);
        if (!$module) {
            return false;
        }

        if (
            strpos($fqn, "{$module}\\Controller\\Admin") === 0
            || strpos($fqn, "{$module}\\Controller\\Customer") === 0
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param Entity $entity
     *
     * @return array
     * @throws EntityException
     */
    private function getInterfaces(Entity $entity): array
    {
        $reflection = $this->getReflection($entity);

        $imports   = $reflection->getImports();
        $namespace = $reflection->getNamespace();

        $result = [];
        foreach ($reflection->getInterfaces() as $interface) {
            $result[] = $this->buildFqn($interface, $imports, $namespace);
        }

        return $result;
    }

    /**
     * @param string $name
     *
     * @return string
     */
    private function getAliasRoot(string $name): string
    {
        $pos = strpos($name, '\\');
        if ($pos === false) {
            return $name;
        }

        return substr($name, 0, $pos);
    }

    /**
     * @param string $name
     * @param array  $imports
     * @param string $namespace
     *
     * @return string
     */
    private function buildFqn(string $name, array $imports, string $namespace): string
    {
        $aliasRoot = $this->getAliasRoot($name);
        if ($aliasRoot) {
            $loweredAliasRoot = strtolower($aliasRoot);
            if (isset($imports[$loweredAliasRoot])) {
                $pos = strpos($name, '\\');

                $name = $pos === false
                    ? $imports[$loweredAliasRoot]
                    : $imports[$loweredAliasRoot] . substr($name, $pos + 1);
            } else {
                $name = $namespace . '\\' . $name;
            }
        }

        return ltrim($name, '\\');
    }

    /**
     * @param Entity $entity
     *
     * @return Reflection
     * @throws EntityException
     */
    private function getReflection(Entity $entity): Reflection
    {
        return $this->reflectionFactory->build($entity);
    }
}
