<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XCart\Extender\Action;

use Doctrine\Common\Annotations\AnnotationException;
use ReflectionException;
use XCart\Extender\Exception\ParserException;
use XCart\Extender\Model\Reflection;

use function count;
use function in_array;
use function is_array;
use function strtolower;
use function token_get_all;

use const T_ABSTRACT;
use const T_AS;
use const T_CLASS;
use const T_DOC_COMMENT;
use const T_EXTENDS;
use const T_FINAL;
use const T_IMPLEMENTS;
use const T_INTERFACE;
use const T_NAMESPACE;
use const T_NS_SEPARATOR;
use const T_STRING;
use const T_TRAIT;
use const T_USE;

if (!defined('T_NAME_QUALIFIED')) {
    define('T_NAME_QUALIFIED', -1);
}

if (!defined('T_NAME_FULLY_QUALIFIED')) {
    define('T_NAME_FULLY_QUALIFIED', -1);
}

class Parser implements ParserInterface
{
    /**
     * @var int
     */
    private int $position = 0;

    /**
     * @var int
     */
    private int $count = 0;

    /**
     * @var AnnotationsParserInterface
     */
    private AnnotationsParserInterface $annotationsParser;

    /**
     * @param AnnotationsParserInterface $annotationsParser
     */
    public function __construct(AnnotationsParserInterface $annotationsParser)
    {
        $this->annotationsParser = $annotationsParser;
    }

    /**
     * @param string $sourceCode
     *
     * @return Reflection
     * @throws ParserException
     */
    public function parseSource(string $sourceCode): Reflection
    {
        $tokens = token_get_all($sourceCode);

        $this->position = 0;
        $this->count    = count($tokens);

        $reflection = new Reflection();

        $reflection->setNamespace($this->getNamespace($tokens));
        $reflection->setImports($this->getImports($tokens));
        $reflection->setDocComment($this->getDocComment($tokens));
        $reflection->setModifier($this->getModifier($tokens));
        $reflection->setKind($this->getKind($tokens));
        $reflection->setName($this->getName($tokens));
        $reflection->setParent($this->getParent($tokens));
        $reflection->setInterfaces($this->getInterfaces($tokens));

        try {
            $reflection->setAnnotations($this->annotationsParser->parseAnnotations($reflection));
        } catch (AnnotationException | ReflectionException $exception) {
            throw ParserException::fromAnnotationsParsing($exception);
        }

        return $reflection;
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @throws ParserException
     */
    private function getNamespace(array $tokens): string
    {
        $result = '';

        $this->seekToAny($tokens, [T_NAMESPACE, T_USE]);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_STRING:
                case T_NS_SEPARATOR:
                case T_NAME_QUALIFIED:
                case T_NAME_FULLY_QUALIFIED:
                    $result .= $content;
                    break;
                case ';':
                case T_USE:
                    return $result;
            }

            $this->position++;
        }

        throw ParserException::fromParsing('namespace');
    }

    /**
     * @param array $tokens
     *
     * @return array
     * @throws ParserException
     */
    private function getImports(array $tokens): array
    {
        $result = [];

        $import = '';
        $alias  = '';

        $index = $this->position;

        $this->seekToAny($tokens, [T_USE, T_CLASS, T_INTERFACE, T_TRAIT]);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_STRING:
                    $import .= $content;
                    $alias  = $content;
                    break;
                case T_NS_SEPARATOR:
                    $import .= $content;
                    break;
                case T_NAME_QUALIFIED:
                case T_NAME_FULLY_QUALIFIED:
                    $import .= $content;
                    $alias = substr(
                        (string) $content,
                        (int) (strrpos((string) $content, '\\')) + 1
                    );
                    break;
                case T_AS:
                    $this->seekToAny($tokens, [T_STRING]);
                    [, $alias] = $this->current($tokens);
                    break;
                case ';':
                case ',':
                    $result[strtolower($alias)] = $import;
                    [$index, $alias, $import] = [$this->position, '', ''];
                    break;
                case T_CLASS:
                case T_INTERFACE:
                case T_TRAIT:
                    $this->position = $index;

                    return $result;
            }

            $this->position++;
        }

        throw ParserException::fromParsing('imports');
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @noinspection PhpInconsistentReturnPointsInspection
     */
    private function getDocComment(array $tokens): string
    {
        $result = '';

        $index = $this->position;

        $this->seekToAny($tokens, [T_DOC_COMMENT, T_NAMESPACE, T_USE, T_CLASS, T_INTERFACE, T_TRAIT]);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_DOC_COMMENT:
                    [$index, $result] = [$this->position, $content];
                    break;
                case T_USE:
                case T_NAMESPACE:
                    [$index, $result] = [$this->position, ''];
                    break;
                case T_CLASS:
                case T_INTERFACE:
                case T_TRAIT:
                    $this->position = $index;

                    return $result;
            }

            $this->position++;
        }
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @noinspection PhpInconsistentReturnPointsInspection
     */
    private function getModifier(array $tokens): string
    {
        $result = '';

        $index = $this->position;
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_FINAL:
                case T_ABSTRACT:
                    [$index, $result] = [$this->position, $content];
                    break;
                case T_CLASS:
                case T_INTERFACE:
                case T_TRAIT:
                    $this->position = $index;

                    return $result;
            }

            $this->position++;
        }
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @noinspection PhpInconsistentReturnPointsInspection
     */
    private function getKind(array $tokens): string
    {
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_CLASS:
                case T_INTERFACE:
                case T_TRAIT:
                    return strtolower($content);
            }

            $this->position++;
        }
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @throws ParserException
     */
    private function getName(array $tokens): string
    {
        $this->seekToAny($tokens, [T_CLASS, T_INTERFACE, T_TRAIT]);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            if ($token === T_STRING) {
                return $content;
            }

            $this->position++;
        }

        throw ParserException::fromParsing('name');
    }

    /**
     * @param array $tokens
     *
     * @return string
     * @throws ParserException
     */
    private function getParent(array $tokens): string
    {
        $result = '';

        $this->seekToAny($tokens, [T_EXTENDS, '{']);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_STRING:
                case T_NS_SEPARATOR:
                case T_NAME_QUALIFIED:
                case T_NAME_FULLY_QUALIFIED:
                    $result .= $content;
                    break;
                case T_IMPLEMENTS:
                case '{':
                    return $result;
            }

            $this->position++;
        }

        throw ParserException::fromParsing('parent');
    }

    /**
     * @param array $tokens
     *
     * @return array
     * @throws ParserException
     */
    private function getInterfaces(array $tokens): array
    {
        $result = [];

        $interface = '';

        $this->seekToAny($tokens, [T_IMPLEMENTS, '{']);
        while ($this->position < $this->count) {
            [$token, $content] = $this->current($tokens);

            switch ($token) {
                case T_STRING:
                case T_NS_SEPARATOR:
                case T_NAME_QUALIFIED:
                case T_NAME_FULLY_QUALIFIED:
                    $interface .= $content;
                    break;
                case ',':
                    $result[]  = $interface;
                    $interface = '';
                    break;
                case '{':
                    if ($interface) {
                        $result[] = $interface;
                    }

                    return $result;
            }

            $this->position++;
        }

        throw ParserException::fromParsing('interfaces');
    }

    /**
     * @param array $tokens
     * @param array $seekToList
     */
    private function seekToAny(array $tokens, array $seekToList): void
    {
        while ($this->position < $this->count) {
            [$token] = $this->current($tokens);
            if (in_array($token, $seekToList, true)) {
                break;
            }

            $this->position++;
        }
    }

    /**
     * @param array $tokens
     *
     * @return array
     */
    private function current(array $tokens): array
    {
        $token = $tokens[$this->position];

        return is_array($token) ? $token : [$token, $token];
    }
}
